package ru.yandex.solomon.tool.stockpile;

import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardWriterTest {

    private int shardId;
    private StockpileShardWriter writer;
    private StockpileClientStub stockpile;

    @Before
    public void setUp() {
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        shardId = stockpile.randomShardId();
        writer = new StockpileShardWriter(shardId, stockpile, ForkJoinPool.commonPool());
    }

    @Test
    public void empty() {
        var future = writer.doneFuture();
        assertFalse(future.isDone());

        writer.complete();
        future.join();

        assertTrue(writer.doneFuture().isDone());
    }

    @Test
    public void writeOne() {
        int mask = TsColumn.mask | ValueColumn.mask;
        Metric metric = randomMetric(mask, 10);

        writer.write(metric).join();
        assertFalse(writer.doneFuture().isDone());
        writer.complete();
        writer.doneFuture().join();

        var result = getArchive(metric.shardId, metric.localId);
        assertEquals(metric.archive, result);
    }

    @Test
    public void writeMany() {
        int mask = TsColumn.mask | ValueColumn.mask;
        List<Metric> metrics = IntStream.range(0, 10_000)
            .parallel()
            .mapToObj(ignore -> randomMetric(mask, ThreadLocalRandom.current().nextInt(1, 500)))
            .collect(Collectors.toList());

        System.out.println("Start write metrics");
        for (Metric metric : metrics) {
            writer.write(metric).join();
            assertFalse(writer.doneFuture().isDone());
        }
        System.out.println("Done write metrics");

        writer.complete();
        writer.doneFuture().join();

        for (Metric metric : metrics) {
            var result = getArchive(metric.shardId, metric.localId);
            assertEquals(metric.archive, result);
        }
    }

    @Test
    public void aggregateMetricsWritesAsNotAggregate() {
        int mask = TsColumn.mask | ValueColumn.mask | CountColumn.mask | MergeColumn.mask;
        var metric = randomMetric(mask, 1000);

        writer.write(metric).join();
        writer.complete();
        writer.doneFuture().join();

        var expected = new MetricArchiveMutable();
        var it = metric.archive.iterator();
        var point = new AggrPoint();
        while (it.next(point)) {
            point.merge = false;
            expected.addRecordData(mask, point);
        }

        var result = getArchive(metric.shardId, metric.localId);
        assertEquals(expected.toImmutableNoCopy(), result);
        close(expected, result);
    }

    private MetricArchiveImmutable getArchive(int shardId, long localId) {
        return stockpile.getTimeSeries(shardId, localId);
    }

    private Metric randomMetric(int mask, int points) {
        var archive = randomArchive(mask, points);
        return new Metric(shardId, StockpileLocalId.random(), archive);
    }

    private MetricArchiveImmutable randomArchive(int mask, int points) {
        try (var archive = new MetricArchiveMutable()) {
            RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
            long tsMillis = TsRandomData.randomTs(ThreadLocalRandom.current());
            for (int index = 0; index < points; index++) {
                tsMillis += 10_000;
                randomPoint(point, mask);
                point.tsMillis = tsMillis;
                archive.addRecordData(mask, point);
            }
            return archive.toImmutableNoCopy();
        }
    }
}
