package ru.yandex.solomon.tool.stockpile.backup;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.springframework.test.util.AssertionErrors.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class BackupsHelperTest {
    private KikimrKvClientInMem kvClient;
    private BackupsHelper helper;
    private long tabletId;
    private long tabletGen;
    private long tx = ThreadLocalRandom.current().nextInt(100, 10_000_000);

    @Before
    public void setUp() {
        kvClient = new KikimrKvClientInMem();
        helper = new BackupsHelper(kvClient);

        tabletId = kvClient.createKvTablet();
        tabletGen = kvClient.incrementGeneration(tabletId, 0).join();
    }

    @Test
    public void lsEmpty() {
        List<BackupWithStats> result = helper.listBackupsWithFiles(tabletId, 0).join();
        assertEquals(Collections.emptyList(), result);
    }

    @Test
    public void backupEmpty() {
        FileNamePrefix.Backup backup = new FileNamePrefix.Backup(Instant.now().getEpochSecond());
        helper.backupCurrent(tabletId, 0, backup).join();

        List<BackupWithStats> result = helper.listBackupsWithFiles(tabletId, 0).join();
        assertEquals(Collections.emptyList(), result);
    }

    @Test
    public void backupEternity() {
        backupLevel(SnapshotLevel.ETERNITY);
    }

    @Test
    public void backupDaily() {
        backupLevel(SnapshotLevel.DAILY);
    }

    @Test
    public void backupTwoHours() {
        backupLevel(SnapshotLevel.TWO_HOURS);
    }

    @Test
    public void rm() {
        for (SnapshotLevel level : SnapshotLevel.values()) {
            syncWriteChunk(level, 1, randomBytes());
            syncWriteChunk(level, 2, randomBytes());

            syncWriteIndex(level, 1, true, randomBytes());
        }

        FileNamePrefix.Backup backup = new FileNamePrefix.Backup(Instant.now().getEpochSecond());
        helper.backupCurrent(tabletId, 0, backup).join();

        List<BackupWithStats> beforeRmLs = helper.listBackupsWithFiles(tabletId, 0).join();
        assertFalse(beforeRmLs.isEmpty());

        helper.deleteBackups(tabletId, 0, backup);

        List<BackupWithStats> afterRmLs = helper.listBackupsWithFiles(tabletId, 0).join();
        Assert.assertTrue(afterRmLs.isEmpty());
    }

    @Test
    public void restoreFromBackupEternity() {
        backupAndRestoreLevel(SnapshotLevel.ETERNITY);
    }

    @Test
    public void restoreFromBackupDaily() {
        backupAndRestoreLevel(SnapshotLevel.DAILY);
    }

    @Test
    public void restoreFromBackupTwoHours() {
        backupAndRestoreLevel(SnapshotLevel.TWO_HOURS);
    }

    private void backupLevel(SnapshotLevel level) {
        syncWriteChunk(level, 1, new byte[]{0x42});
        syncWriteChunk(level, 2, new byte[]{0x43});
        syncWriteChunk(level, 3, new byte[]{0x44});

        syncWriteIndex(level, 1, true, new byte[]{0x13, 0x14, 0x15});

        FileNamePrefix.Backup backup = new FileNamePrefix.Backup(Instant.now().getEpochSecond());
        helper.backupCurrent(tabletId, 0, backup).join();

        List<String> backupFiles = helper.listBackupsWithFiles(tabletId, 0)
                .join()
                .stream()
                .flatMap(stats -> stats.getFiles().stream())
                .map(KikimrKvClient.KvEntryStats::getName)
                .collect(Collectors.toList());

        String message = backupFiles.toString();
        assertTrue(message, backupFiles.contains(nameOfChunk(level, 1, backup)));
        assertTrue(message, backupFiles.contains(nameOfChunk(level, 2, backup)));
        assertTrue(message, backupFiles.contains(nameOfChunk(level, 3, backup)));

        assertTrue(message, backupFiles.contains(nameOfIndex(level, 1, true, backup)));
    }

    private void backupAndRestoreLevel(SnapshotLevel level) {
        Map<String, byte[]> filesV1 = randomFiles(level, 10);
        syncWrite(filesV1);

        FileNamePrefix.Backup backup = new FileNamePrefix.Backup(Instant.now().getEpochSecond());
        helper.backupCurrent(tabletId, 0, backup).join();

        // old files removes after some times
        kvClient.deleteRange(tabletId, tabletGen, StockpileKvNames.currentFilesRange(), 0).join();

        List<String> filesAfterBackup = new ArrayList<>();
        for (int index = 0; index < 3; index++) {
            tx++;
            Map<String, byte[]> data = randomFiles(level, 5);
            syncWrite(data);
            filesAfterBackup.addAll(data.keySet());
        }

        // restore
        helper.restoreBackup(tabletId, backup).join();

        tabletGen = kvClient.incrementGeneration(tabletId, 0).join();
        Map<String, byte[]> fileToBytes = Stream.of(kvClient.readRange(tabletId, tabletGen, StockpileKvNames.currentFilesRange(), true, 0)
                .join()
                .getEntriesAll())
                .collect(Collectors.toMap(KikimrKvClient.KvEntryWithStats::getName, KikimrKvClient.KvEntryWithStats::getValue));

        for (String name : filesAfterBackup) {
            assertFalse(fileToBytes.keySet().toString(), fileToBytes.containsKey(name));
        }

        for (String name : filesV1.keySet()) {
            byte[] expected = filesV1.get(name);
            byte[] restored = fileToBytes.get(name);
            assertArrayEquals(name, expected, restored);
        }
    }

    private Map<String, byte[]> randomFiles(SnapshotLevel level, int max) {
        final int maxChunks = ThreadLocalRandom.current().nextInt(1, max);
        final int maxIndexes = ThreadLocalRandom.current().nextInt(1, max);

        Map<String, byte[]> result = new LinkedHashMap<>();
        for (int index = 0; index < maxChunks; index++) {
            result.put(nameOfChunk(level, index), randomBytes());
        }

        for (int index = 0; index < maxIndexes; index++) {
            result.put(nameOfIndex(level, index, index + 1 == maxIndexes), randomBytes());
        }

        return result;
    }

    private void syncWriteChunk(SnapshotLevel level, int index, byte[] value) {
        syncWrite(nameOfChunk(level, index), value);
    }

    private void syncWriteIndex(SnapshotLevel level, int index, boolean last, byte[] value) {
        syncWrite(nameOfIndex(level, index, last), value);
    }

    private String nameOfChunk(SnapshotLevel level, int index) {
        return nameOfChunk(level, index, FileNamePrefix.Current.instance);
    }

    private String nameOfChunk(SnapshotLevel level, int index, FileNamePrefix prefix) {
        return StockpileKvNames.chunkFileName(level, tx, index, prefix);

    }

    private String nameOfIndex(SnapshotLevel level, int index, boolean last) {
        return nameOfIndex(level, index, last, FileNamePrefix.Current.instance);
    }

    private String nameOfIndex(SnapshotLevel level, int index, boolean last, FileNamePrefix prefix) {
        return new IndexFile(level, tx, index, last).reconstruct(prefix);
    }

    private void syncWrite(Map<String, byte[]> files) {
        files.forEach(this::syncWrite);
    }

    private void syncWrite(String key, byte[] value) {
        kvClient.write(tabletId, tabletGen, key, value, MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND, 0).join();
    }

    private byte[] randomBytes() {
        int size = ThreadLocalRandom.current().nextInt(1, 1 << 5);
        byte[] result = new byte[size];
        ThreadLocalRandom.current().nextBytes(result);
        return result;
    }
}
