#include <solomon/libs/cpp/multi_shard/multi_shard.h>
#include <solomon/libs/cpp/http/client/curl/client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/text/text.h>
#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/ptr.h>
#include <util/stream/file.h>
#include <util/stream/input.h>
#include <util/stream/mem.h>
#include <util/string/builder.h>

using namespace NSolomon;
using namespace NSolomon::NMultiShard;
using namespace NMonitoring;
using namespace NThreading;


IOutputStream& operator<<(IOutputStream& os, const THeader& header) {
    os << "Version: " << (ui32)header.FormatVersion << '\n'
       << "Continuation Token: " << header.ContinuationToken << '\n';

    return os;
}

class TMultiShardMessagePrinter: public IMessageHandler {
public:
    TMultiShardMessagePrinter(IOutputStream& out, EFormat outputFormat, bool onlyNames)
        : Out_{out}
        , OnlyNames_{onlyNames}
        , Format_{outputFormat}
    {
    }

    bool OnHeader(THeader header) override {
        Out_ << "Header:\n" << header << '\n';
        return true;
    }

    bool OnShardData(TString project, TString cluster, TString service, TString data) override {
        Out_ << "-----------------"
            << "Shard: " << project << '/' << cluster << '/' << service << '\n';

        if (OnlyNames_) {
            return true;
        }

        TMemoryInput in{data};
        IMetricEncoderPtr encoder;

        switch (Format_) {
        case EFormat::JSON:
            encoder = BufferedEncoderJson(&Out_, 4);
            break;
        case EFormat::TEXT:
            encoder = EncoderText(&Out_);
            break;
        default:
            Y_FAIL("Unsupported data type");
        };

        try {
            DecodeSpackV1(&in, encoder.Get());
        } catch (...) {
            Cerr << "Error decoding message data: " << CurrentExceptionMessage() << Endl;
            return false;
        }

        return true;
    }

    void OnError(TString msg) override {
        Cerr << "Error reported by multishard decoder: " << msg << Endl;
    }

    void OnStreamEnd() override {
    }

private:
    IOutputStream& Out_;
    bool OnlyNames_{false};
    EFormat Format_{EFormat::JSON};
};

int main(int argc, const char** argv) {
    using namespace NLastGetopt;

    bool onlyNames{false};

    TOpts opts;
    opts.AddLongOption('o', "out", "write output into the specified file")
        .RequiredArgument("FILE")
        .Optional();

    opts.AddLongOption('f', "format", "output format")
        .RequiredArgument("FORMAT")
        .DefaultValue("JSON")
        .Optional();

    opts.AddLongOption('s', "shard-names", "output only project/cluster/service for each shard")
        .NoArgument()
        .SetFlag(&onlyNames)
        .Optional();

    opts.AddHelpOption('h');
    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "<host:port>" "host and port to connect to");

    TOptsParseResult r{&opts, argc, argv};

    const auto hostPort = TString{r.GetFreeArgs()[0]};

    NMonitoring::TMetricRegistry registry;
    auto httpClient = CreateCurlClient({}, registry);

    TStringBuilder sb;
    sb << hostPort << "/storage/readAll";
    TString url = sb;

    auto p = NewPromise<IHttpClient::TResult>();
    auto cb = [p] (auto result) mutable {
        p.SetValue(std::move(result));
    };

    httpClient->Request(
        Post(sb << "?format=SPACK", {}, Headers()),
        cb
    );

    auto result = p.GetFuture().ExtractValueSync();

    if (result.Fail()) {
        Cerr << "Request to " << url << "failed: " << result.Error().Message() << Endl;
        return 1;
    }

    auto response = result.Extract();
    auto data = response->Data();

    THolder<IOutputStream> output;
    if (r.Has("out")) {
        output.Reset(new TFileOutput{r.Get("out")});
    }

    auto format = FromString<EFormat>(r.Get("format"));

    TMultiShardMessagePrinter printer{output ? *output : Cout, format, onlyNames};
    auto d = CreateMultiShardContinuousChunkDecoder(printer);
    d->Decode(data);

    return 0;
}
