package main

import (
	"bufio"
	"encoding/binary"
	"fmt"
	"github.com/google/gopacket"
	"github.com/google/gopacket/layers"
	"github.com/google/gopacket/pcap"
	"log"
	"net/http"
	"regexp"
	"strings"
	"time"
)

const (
	SnapLen = 262144
)

func Capture(local *Local, settings *Settings) (chan *http.Request, error) {
	matcher := regexp.MustCompile(local.RegExp)
	if local.File == "" && local.Interface == "" {
		return nil, fmt.Errorf("neither --file nor --iface is given")
	}
	var handle *pcap.Handle
	var err error
	if local.Interface != "" {
		handle, err = pcap.OpenLive(local.Interface, SnapLen, false, 1*time.Millisecond)
	} else {
		handle, err = pcap.OpenOffline(local.File)
	}
	if err != nil {
		return nil, err
	}
	if handle.LinkType() != layers.LinkTypeEthernet {
		return nil, fmt.Errorf("only ethernet link type is supported")
	}
	httpFilter := makeFilterFor(local.Port, handle.SnapLen(), []byte(local.HTTPPrefix))
	if err := handle.SetBPFInstructionFilter(httpFilter); err != nil {
		return nil, err
	}

	packetSource := gopacket.NewPacketSource(handle, handle.LinkType())
	stream := make(chan *http.Request)
	go func(packetSource *gopacket.PacketSource, stream chan<- *http.Request, matcher *regexp.Regexp) {
		capturedPackets := 0
		for packet := range packetSource.Packets() {
			transportLayer := packet.Layer(layers.LayerTypeTCP)
			httpReq, err := tryToParseAsHTTP(string(transportLayer.LayerPayload()), matcher)
			if err != nil {
				log.Printf("failed to parse HTTP: %v\n", err)
			} else if httpReq != nil {
				stream <- httpReq
				capturedPackets++
				if capturedPackets == settings.MaxPackets {
					break
				}
			}
		}
		close(stream)
	}(packetSource, stream, matcher)

	return stream, nil
}

func tryToParseAsHTTP(s string, r *regexp.Regexp) (*http.Request, error) {
	req, err := http.ReadRequest(bufio.NewReader(strings.NewReader(s)))
	if err != nil {
		return nil, err
	}

	if !r.MatchString(req.URL.Path) {
		return nil, nil
	}

	req.Header.Set("Connection", "close")
	req.Header.Set("User-Agent", "pcap-mirror/go")
	req.Header.Del("X-Forwarded-For")
	req.Header.Del("X-Forwarded-Proto")
	req.Header.Del("X-Real-Ip")
	req.Header.Del("Accept")
	req.Header.Del("Accept-Encoding")
	return req, nil
}

const (
	EthernetTypeOffset uint32 = 0x0c
	EthernetHeaderSize uint32 = 0x0e

	IPv4Type uint32 = 0x0800
	IPv6Type uint32 = 0x86dd

	IPv6NextHeaderOffset uint32 = 0x04
	IPv6HeaderSize       uint32 = 0x28

	IPv4FlagsOffset    uint32 = 0x06
	IPv4ProtocolOffset uint32 = 0x09

	TCPProtocolNumber uint32 = 0x06

	TCPPacketDstPortOffset uint32 = 0x02
	TCPPacketDataOffset    uint32 = 0x0c
)

type Instruction struct {
	Opcode    uint16
	JumpTrue  func() uint8
	JumpFalse func() uint8
	Argument  uint32
}

type Program struct {
	Code []Instruction
}

func (p *Program) rel(targetInstruction *int) func() uint8 {
	if targetInstruction == nil {
		return func() uint8 {
			return 0
		}
	}
	nextInstr := len(p.Code) + 1
	return func() uint8 {
		diff := *targetInstruction - nextInstr
		if diff < 0 || diff > 255 {
			log.Fatalln("relative offset out of bounds")
		}
		return uint8(diff)
	}
}

func (p *Program) append(code uint16, jt, jf *int, k uint32) int {
	p.Code = append(p.Code, Instruction{Opcode: code, JumpTrue: p.rel(jt), JumpFalse: p.rel(jf), Argument: k})
	return len(p.Code) - 1
}

func (p *Program) assemble() []pcap.BPFInstruction {
	var code = make([]pcap.BPFInstruction, len(p.Code))

	for i, instr := range p.Code {
		code[i] = pcap.BPFInstruction{
			Code: instr.Opcode,
			Jt:   instr.JumpTrue(),
			Jf:   instr.JumpFalse(),
			K:    instr.Argument,
		}
	}

	return code
}

func makeFilterFor(port uint16, snapLen int, prefix []byte) []pcap.BPFInstruction {
	// pcap bpf compiler sucks at common subexpression elimination, so we do it manually
	var prog = Program{Code: make([]Instruction, 0, 100)}

	// pck[0x0c] is EtherType, load uint16 to reg a
	// 0: ldh [0x0c]
	prog.append(BpfLda|BpfH|BpfAbs, nil, nil, EthernetTypeOffset)

	// EtherType may be IPv4 (0x0800) or IPv6 (0x86dd)
	// 1: jeq 0x86dd, true -> continue, false -> jump to check IPv4
	checkIPv4Label := new(int)
	prog.append(BpfJmp|BpfJeq, nil, checkIPv4Label, IPv6Type)

	// Store IP header size in X
	// 2: ldxi 0x28
	prog.append(BpfLdx|BpfImm, nil, nil, IPv6HeaderSize)

	// Load NextHeader field from IPv6 packet, offset 6 from Ethernet header (which itself is 0x0e bytes)
	// 3: ldb [0x14]
	prog.append(BpfLda|BpfB|BpfAbs, nil, nil, EthernetHeaderSize+IPv6NextHeaderOffset)

	// Compare NextHeader value with TCP protocol number (0x06)
	// 4: jeq 0x06, true -> check tcp packet, false -> skip packet
	tcpPacketCheckLabel := new(int)
	skipPacketLabel := new(int)
	prog.append(BpfJmp|BpfJeq, tcpPacketCheckLabel, skipPacketLabel, TCPProtocolNumber)

	// Packet is not IPv6, try IPv4. EtherType is still in reg a
	// 5: jeq 0x0800, true -> continue, false -> skip packet
	*checkIPv4Label = prog.append(BpfJmp|BpfJeq, nil, skipPacketLabel, IPv4Type)

	// Packet is IPv4. Protocol is at offset 0x09 after Ethernet header, total offset is 0x17
	// 6: ldb [0x17]
	prog.append(BpfLda|BpfB|BpfAbs, nil, nil, EthernetHeaderSize+IPv4ProtocolOffset)

	// Compare Protocol value with TCP protocol number (0x06)
	// 7: jeq 0x06, true -> continue, false -> skip packet
	prog.append(BpfJmp|BpfJeq, nil, skipPacketLabel, TCPProtocolNumber)

	// Check that IP packet is not fragmented. Load Flags | FragmentOffset = offset 6 + 0x0e
	// 8: ldh [0x14]
	prog.append(BpfLda|BpfH|BpfAbs, nil, nil, EthernetHeaderSize+IPv4FlagsOffset)

	// Check that FragmentOffset is zero
	// 9: jset 0x1fff, true -> skip packet, false -> continue
	prog.append(BpfJmp|BpfJset, skipPacketLabel, nil, 0x1fff)

	// Packet is TCP. We need to calculate offset of the TCP packet, for which we need Internet Header Length.
	// IHL is lower 4 bits of first byte of the IPv4 packet, i.e. offset 0x0e from the beginning
	// 10: ldxb 4*([0xe]&0xf)
	prog.append(BpfLdx|BpfB|BpfMsh, nil, nil, EthernetHeaderSize)

	// Load TCP packet dst port to A. IP header size is stored in X, Ethernet header size is fixed 0x0e
	// 11: ldh [x + 0x0e + 0x02]
	*tcpPacketCheckLabel = prog.append(BpfLda|BpfH|BpfInd, nil, nil, EthernetHeaderSize+TCPPacketDstPortOffset)

	// Check if TCP packet dst port matches
	// 12: jeq <dport>, true -> process further, false -> skip packet (continue)
	processTCPPayload := new(int)
	prog.append(BpfJmp|BpfJeq, processTCPPayload, nil, uint32(port))

	// Skip packet fast return
	// 13: ret 0
	*skipPacketLabel = prog.append(BpfRet, nil, nil, 0x0)

	// Find TCP payload data
	// 14: ldb [x + 0x0e + 0x0c]
	*processTCPPayload = prog.append(BpfLda|BpfB|BpfInd, nil, nil, EthernetHeaderSize+TCPPacketDataOffset)

	// 15: and 0xf0
	prog.append(BpfAlu|BpfAnd, nil, nil, 0xf0)

	// 16: rsh 0x02
	prog.append(BpfAlu|BpfRsh, nil, nil, 0x02)

	// X stores IP header length
	// 17: add x
	prog.append(BpfAlu|BpfAdd|BpfX, nil, nil, 0x00)

	// Now A stores IP header length + TCP header length. Move the result to X
	// 18: tax
	prog.append(BpfMisc|BpfTax, nil, nil, 0x00)

	// TCP data is accessible by [X + EthernetHeaderSize + offset]
	retFail := new(int)

	offset := 0
	for offset+4 <= len(prefix) {
		word := prefix[offset : offset+4]

		// ldw [X + EthernetHeaderSize + offset] == tcp.data[offset:4]
		prog.append(BpfLda|BpfW|BpfInd, nil, nil, EthernetHeaderSize+uint32(offset))
		// jeq <word>, true -> continue, false -> skip packet
		prog.append(BpfJmp|BpfJeq, nil, retFail, binary.BigEndian.Uint32(word))

		offset += 4
	}

	if offset+2 <= len(prefix) {
		hword := prefix[offset : offset+2]

		// ldh [X + EthernetHeaderSize + offset] == tcp.data[offset:2]
		prog.append(BpfLda|BpfH|BpfInd, nil, nil, EthernetHeaderSize+uint32(offset))
		// jeq <hword>, true -> continue, false -> skip packet
		prog.append(BpfJmp|BpfJeq, nil, retFail, uint32(binary.BigEndian.Uint16(hword)))

		offset += 2
	}

	if offset < len(prefix) {
		single := prefix[offset]

		// ldb [X + EthernetHeaderSize + offset] == tcp.data[offset:1]
		prog.append(BpfLda|BpfB|BpfInd, nil, nil, EthernetHeaderSize+uint32(offset))
		// jeq <byte>, true -> continue, false -> skip packet
		prog.append(BpfJmp|BpfJeq, nil, retFail, uint32(single))

		offset += 1
	}

	if offset != len(prefix) {
		log.Fatalln("string length mismatch, bug in the code")
	}

	prog.append(BpfRet, nil, nil, uint32(snapLen))
	*retFail = prog.append(BpfRet, nil, nil, 0x0)

	return prog.assemble()
}
