package main

import (
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"bytes"
	"context"
	"encoding/json"
	"github.com/spf13/cobra"
	"log"
	"os"
)

type Local struct {
	Interface  string
	File       string
	Port       uint16
	HTTPPrefix string
	RegExp     string
}

type Remote struct {
	Address string
	Token   string
}

type Settings struct {
	MaxConcurrent int
	MaxPackets    int
}

func main() {
	local := Local{
		Interface:  "",
		File:       "",
		Port:       0,
		HTTPPrefix: "",
		RegExp:     "",
	}

	remote := Remote{
		Address: "",
		Token:   "",
	}

	settings := Settings{
		MaxConcurrent: 0,
		MaxPackets:    0,
	}

	cmd := &cobra.Command{
		Use:          os.Args[0],
		Short:        "Capture traffic and mirror it to another server",
		Args:         cobra.ExactArgs(0),
		SilenceUsage: true,
		Run: func(cmd *cobra.Command, args []string) {
			registry := solomon.NewRegistry(&solomon.RegistryOpts{})
			capture, err := Capture(&local, &settings)
			if err != nil {
				log.Fatalln(err)
			}
			err = Sender(&remote, &settings, capture, registry)
			if err != nil {
				log.Fatalln(err)
			}
			buf := new(bytes.Buffer)
			_, err = registry.StreamJSON(context.Background(), buf)
			if err != nil {
				log.Fatalln(err)
			}
			outBuf := new(bytes.Buffer)
			err = json.Indent(outBuf, buf.Bytes(), "", "  ")
			if err != nil {
				log.Fatalln(err)
			}
			log.Println(outBuf.String())
		},
	}

	cmd.Flags().StringVar(&local.Interface, "iface", "", "local interface to listen")
	cmd.Flags().StringVar(&local.File, "file", "", "file with capture")
	cmd.Flags().Uint16Var(&local.Port, "port", 0, "packet destination port filter")
	_ = cmd.MarkFlagRequired("port")
	cmd.Flags().StringVar(&local.HTTPPrefix, "http-prefix", "",
		"HTTP packet beginning to filter, e.g. POST /foo/bar")
	_ = cmd.MarkFlagRequired("http-prefix")
	cmd.Flags().StringVar(&local.RegExp, "path-regexp", "",
		"Additional regexp to filter requests by path, eq. /api/")

	cmd.Flags().StringVar(&remote.Address, "remote", "", "remote address to send request to")
	_ = cmd.MarkFlagRequired("remote")

	cmd.Flags().IntVar(&settings.MaxConcurrent, "max-concurrent", 10,
		"max number of concurrent http requests")
	cmd.Flags().IntVar(&settings.MaxPackets, "max-packets", 0,
		"stop after specific number of packets captured")

	if value, ok := os.LookupEnv("SOLOMON_TOKEN"); ok {
		remote.Token = value
	}

	if err := cmd.Execute(); err != nil {
		os.Exit(1)
	}
}
