package main

import (
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"net/http"
	"strconv"
	"sync"
	"time"
)

func Sender(remote *Remote, settings *Settings, pipe <-chan *http.Request, registry *solomon.Registry) error {
	t := http.DefaultTransport.(*http.Transport).Clone()
	t.MaxIdleConns = settings.MaxConcurrent
	t.MaxIdleConnsPerHost = settings.MaxConcurrent
	t.MaxIdleConnsPerHost = settings.MaxConcurrent

	mold, err := http.NewRequest("GET", remote.Address, nil)
	if err != nil {
		return err
	}
	if remote.Token != "" {
		mold.Header.Set("Authorization", "OAuth "+remote.Token)
	}

	httpClient := &http.Client{
		Timeout:   30 * time.Second,
		Transport: t,
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			return http.ErrUseLastResponse
		},
	}

	semaphoreChan := make(chan struct{}, settings.MaxConcurrent)

	defer func() {
		close(semaphoreChan)
	}()

	overflow := registry.Counter("sender.overflow")
	solomon.Rated(overflow)
	failures := registry.Counter("sender.response.failure")
	solomon.Rated(failures)
	statuses := registry.CounterVec("sender.response.status", []string{"code"})
	solomon.Rated(statuses)

	wg := sync.WaitGroup{}
	for req := range pipe {
		wg.Add(1)
		go func(req *http.Request, mold *http.Request, httpClient *http.Client) {
			defer wg.Done()
			select {
			case semaphoreChan <- struct{}{}:
			default:
				overflow.Inc()
				// overflow, skip
				return
			}

			req.Host = mold.Host
			req.URL.Scheme = mold.URL.Scheme
			req.URL.Host = mold.URL.Host
			req.RequestURI = ""

			auth := mold.Header.Get("Authorization")
			if auth != "" {
				req.Header.Set("Authorization", auth)
			}

			//req.URL = baseUrl + req.URL
			resp, err := httpClient.Do(req)
			if err != nil {
				failures.Inc()
			} else {
				statuses.With(map[string]string{"code": strconv.Itoa(resp.StatusCode)}).Inc()
				//body, err := ioutil.ReadAll(resp.Body)
				//if err == nil {
				//	fmt.Printf("%s %s\n", resp.Status, string(body))
				//}
			}

			<-semaphoreChan
		}(req, mold, httpClient)
	}

	wg.Wait()

	return nil
}
