package cmd

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/svn"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"os"
	"path/filepath"
	"sort"
	"strconv"
	"strings"
	"time"
)

func init() {
	branchCmd := &cobra.Command{
		Use:   "branch",
		Short: "Commands to manipulate release branches",
	}
	forkCmd := &cobra.Command{
		Use:   "fork [revision]",
		Short: "Create release branch from the latest commit in arcadia/solomon or specified revision",
		Args:  cobra.MaximumNArgs(1),
		RunE:  runForkCmd,
	}
	listCmd := &cobra.Command{
		Use:   "list [limit]",
		Short: "List release branches",
		Args:  cobra.MaximumNArgs(1),
		RunE:  runListCmd,
	}
	removeCmd := &cobra.Command{
		Use:   "remove <name>",
		Short: "Remove specified release branch",
		Args:  cobra.ExactArgs(1),
		RunE:  runRemoveCmd,
	}
	mergeCmd := &cobra.Command{
		Use:   "merge [name] revision1 [revision2 ...]",
		Short: "Merge given revisions into specified branch or the latest branch",
		Args:  cobra.MinimumNArgs(1),
		RunE:  runMergeCmd,
	}

	branchCmd.AddCommand(forkCmd)
	branchCmd.AddCommand(listCmd)
	branchCmd.AddCommand(removeCmd)
	branchCmd.AddCommand(mergeCmd)
	rootCmd.AddCommand(branchCmd)
}

func getForkRevision(ctx context.Context, args []string) (int, error) {
	if len(args) == 0 {
		revision, err := svn.GetLatestRevision(ctx, "trunk", "/solomon")
		if err != nil {
			return 0, fmt.Errorf("cannot get lastest revision of /solomon: %w", err)
		}
		return revision, nil
	}

	// remove first 'r' from revision
	revisionStr := args[0]
	if revisionStr[0] == 'r' {
		revisionStr = revisionStr[1:]
	}
	revision, err := strconv.Atoi(revisionStr)
	if err != nil {
		return 0, fmt.Errorf("cannot parse revision from '%s'", revisionStr)
	}
	return revision, nil
}

func runForkCmd(cmd *cobra.Command, args []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()

	revision, err := getForkRevision(ctx, args)
	if err != nil {
		return err
	}

	releaseBranch := "stable-" + time.Now().Format("2006-01-02")
	log.Printf("fork release branch %s from revision r%d\n", releaseBranch, revision)

	err = svn.ForkReleaseBranch(ctx, releaseBranch, revision)
	if err != nil {
		return err
	}

	log.Println(color.Green("Ok"))
	log.Printf("Branch history: https://a.yandex-team.ru/arc/history/branches/solomon/%s", releaseBranch)
	return nil
}

func runListCmd(cmd *cobra.Command, args []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()

	branches, err := svn.ListReleaseBranches(ctx)
	if err != nil {
		return fmt.Errorf("cannot list solomon release branches: %w", err)
	}

	if len(args) > 0 {
		limit, err := strconv.Atoi(args[0])
		if err != nil {
			return fmt.Errorf("could not parse limit argument: %w", err)
		}
		if limit < len(branches) {
			branches = branches[len(branches)-limit:]
		}
	}
	for _, branch := range branches {
		fmt.Println(branch)
	}
	return nil
}

func runRemoveCmd(cmd *cobra.Command, args []string) error {
	branchName := args[0]
	if !strings.HasPrefix(branchName, "stable-") {
		return fmt.Errorf("invalid branch name '%s'", branchName)
	}

	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()

	err := svn.RemoveReleaseBranch(ctx, branchName)
	if err != nil {
		return fmt.Errorf("cannot remove solomon release branch: %w", err)
	}

	log.Println(color.Green("Ok"))
	return nil
}

func parseRevisions(revisionStrings []string) ([]int, error) {
	revisionsMap := make(map[int]bool)
	for _, rev := range revisionStrings {
		// also support r123456 format
		revNum, err := strconv.Atoi(strings.TrimPrefix(rev, "r"))
		if err != nil {
			return nil, fmt.Errorf("invalid revision number: '%s'", rev)
		}
		if revisionsMap[revNum] {
			return nil, fmt.Errorf("duplicate revision number: %d", revNum)
		}
		revisionsMap[revNum] = true
	}

	revisions := make([]int, 0, len(revisionsMap))
	for rev := range revisionsMap {
		revisions = append(revisions, rev)
	}

	sort.Ints(revisions)
	return revisions, nil
}

func toString(nums []int) string {
	strs := make([]string, 0, len(nums))
	for _, num := range nums {
		strs = append(strs, "r"+strconv.Itoa(num))
	}
	return strings.Join(strs, ", ")
}

func prepareLocalCopy(branchName string) (string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Minute)
	defer cancel()

	stableDir := filepath.Join(os.Getenv("HOME"), ".solomon", "stable")
	branchDir := filepath.Join(stableDir, branchName)

	if _, err := os.Stat(branchDir); os.IsNotExist(err) {
		// create parent dir if needed
		if _, err := os.Stat(stableDir); os.IsNotExist(err) {
			log.Printf("Create dir %s\n", stableDir)
			if err = os.MkdirAll(stableDir, os.ModePerm); err != nil {
				return "", fmt.Errorf("cannot create dir %s: %w", stableDir, err)
			}
		}

		log.Printf("Checkout branch %s in %s\n", branchName, branchDir)
		// here we to use parent dir, because svn will automatically create proper subdir for branch
		if err = svn.CheckoutReleaseBranch(ctx, branchName, stableDir); err != nil {
			return "", fmt.Errorf("cannot checkout release branch %s in %s: %w", branchName, branchDir, err)
		}
	} else {
		log.Printf("Update local branch copy in %s\n", branchDir)
		if err = svn.Update(ctx, branchDir); err != nil {
			return "", fmt.Errorf("cannot update release branch in %s, %w", branchDir, err)
		}
	}
	return branchDir, nil
}

func runMergeCmd(cmd *cobra.Command, args []string) error {
	var branchName string
	if strings.HasPrefix(args[0], "stable-") {
		branchName = args[0]
		args = args[1:]
	} else {
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
		defer cancel()

		branches, err := svn.ListReleaseBranches(ctx)
		if err != nil {
			return fmt.Errorf("cannot list solomon release branches: %w", err)
		}

		if len(branches) == 0 {
			return fmt.Errorf("there is no stable branches yet")
		}

		branchName = branches[len(branches)-1]
	}

	revisions, err := parseRevisions(args)
	if err != nil {
		return err
	}

	branchDir, err := prepareLocalCopy(branchName)
	if err != nil {
		return err
	}

	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*2)
	defer cancel()

	revisionsList := toString(revisions)
	log.Printf("Merging revisions {%s} into %s\n", revisionsList, branchDir)
	if err := svn.MergeFromTrunk(ctx, branchDir, revisions); err != nil {
		return fmt.Errorf("cannot merge commits into release branch %s: %w", branchDir, err)
	}

	if status, err := svn.PrintStatus(ctx, branchDir); err != nil {
		log.Printf("cannot print svn status: %s", err.Error())
	} else {
		log.Printf("svn status:\n%s", status)
	}

	if cli.Confirm("Commit release branch? [yes/No]") {
		log.Printf("Commit changes from %s\n", branchDir)
		commitRev, err := svn.Commit(ctx, branchDir, "Merge from trunk: "+revisionsList)
		if err != nil {
			return fmt.Errorf("cannot commit release branch %s: %w", branchDir, err)
		}
		log.Printf(color.Green("Check commit https://a.yandex-team.ru/arc/commit/%d\n"), commitRev)
	} else {
		log.Println(color.BoldYellow("Changes were not committed"))
	}

	return nil
}
