package cmd

import (
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/infra"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"strings"
	"time"
)

func init() {
	createEventCmd := &cobra.Command{
		Use:   "create-event",
		Short: "Create maintenance event in Infra",
		RunE:  runCreateEventCmd,
	}

	flags := createEventCmd.Flags()

	// required flags
	flags.String("service", "", "service name, valid values are: ["+strings.Join(infra.ListServices(), ", ")+"]")
	_ = createEventCmd.MarkFlagRequired("service")
	flags.String("env", "", "service environment name, valid values are: ["+strings.Join(hosts.EnvNames[1:], ", ")+"]")
	_ = createEventCmd.MarkFlagRequired("env")
	flags.String("title", "", "event title")
	_ = createEventCmd.MarkFlagRequired("title")
	flags.String("description", "", "event description")
	_ = createEventCmd.MarkFlagRequired("description")

	// optional flags
	flags.String("duration", "30m", "planned duration of the event")
	flags.String("ticket", "", "ticket identifier")
	flags.String("dc", "", "datacenter name, valid values are: [man, vla, sas, myt, iva]")

	rootCmd.AddCommand(createEventCmd)
}

func runCreateEventCmd(cmd *cobra.Command, args []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}

	flags := cmd.Flags()
	service, _ := flags.GetString("service")
	envStr, _ := flags.GetString("env")
	title, _ := flags.GetString("title")
	description, _ := flags.GetString("description")
	durationStr, _ := flags.GetString("duration")
	ticket, _ := flags.GetString("ticket")
	dcsStr, _ := flags.GetString("dc")

	env := hosts.EnvFromStr(envStr)
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment '%s'", envStr)
	}

	serviceEnv, err := infra.FindServiceEnv(service, env)
	if err != nil {
		return fmt.Errorf("cannot find service environment: %w", err)
	}

	duration, err := time.ParseDuration(durationStr)
	if err != nil {
		return fmt.Errorf("cannot parse duration string '%s': %v", duration, err)
	}

	dcs, err := hosts.ParseDcs(dcsStr)
	if err != nil {
		return err
	}

	event := serviceEnv.NewEvent(title, description, duration, ticket, dcs)

	client := infra.NewClient(token)
	if err = client.CreateEvent(ctx, event); err != nil {
		return err
	}

	fmt.Printf("created event: %s\n", event.URL())
	return nil
}
