package cmd

import (
	"context"
	"fmt"
	"log"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/solomon/libs/go/brexp"

	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/sandbox"
	"a.yandex-team.ru/solomon/tools/release/internal/svn"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"

	"github.com/spf13/cobra"
)

// ==========================================================================================

func init() {
	packageCmd := &cobra.Command{
		Use:   "package",
		Short: "build and/or deploy packages",
	}
	buildCmd := &cobra.Command{
		Use:   "build",
		Short: "build packages",
		Args:  cobra.NoArgs,
		RunE:  runBuildPackageCmd,
	}
	deployCmd := &cobra.Command{
		Use:   "deploy",
		Short: "deploy packages",
		Args:  cobra.NoArgs,
		RunE:  runDeployPackageCmd,
	}
	buildDeployCmd := &cobra.Command{
		Use:   "builddeploy",
		Short: "build and deploy packages",
		Args:  cobra.NoArgs,
		RunE:  runBuildAndDeployPackageCmd,
	}
	packageCmd.AddCommand(buildCmd)
	packageCmd.AddCommand(deployCmd)
	packageCmd.AddCommand(buildDeployCmd)
	rootCmd.AddCommand(packageCmd)

	initBuildFlags(buildCmd, "pkg-paths")
	initDeployFlags(deployCmd, "z2-configs", "packages")

	initBuildFlags(buildDeployCmd, "pkg-paths", "repos")
	initDeployFlags(buildDeployCmd, "z2-configs")
}

func initBuildFlags(comm *cobra.Command, reqs ...string) {
	fs := comm.Flags()

	_ = fs.StringP("branch", "b", "trunk", "branch to use for package build")
	_ = fs.StringP("revision", "r", "latest", "svn revision")
	_ = fs.StringP("pkg-paths", "p", "", "paths to pkg.json files or package names in solomon/packages")
	_ = fs.String("changelog", "", "changelog")
	_ = fs.BoolP("strip", "s", false, "strip binaries")
	_ = fs.BoolP("full-strip", "f", false, "full strip binaries")
	_ = fs.StringP("repos", "o", "", "repos to publish packages (by default do not publish)")
	_ = fs.StringP("distribution", "d", "unstable", "distribution to publish packages")
	_ = fs.IntP("dupload-attempts", "u", 3, "max dupload attempts")
	_ = fs.String("platform", "linux", "platform")
	_ = fs.Int("memory", 4, "memory to request, Gb")
	_ = fs.Int("disk", 50, "disk to request, Gb")
	_ = fs.Duration("timeout", time.Hour, "task timeout")
	_ = fs.IntP("max-tasks", "t", 5, "maximum tasks to build packages (hard limit)")
	_ = fs.IntP("max-pkgs", "k", 5, "maximum packages per one task to build")
	_ = fs.String("yt-token", "", "token for yt build cache in yav")

	for _, req := range reqs {
		_ = comm.MarkFlagRequired(req)
	}
}

func initDeployFlags(comm *cobra.Command, reqs ...string) {
	fs := comm.Flags()

	_ = fs.StringP("z2-configs", "c", "", "Z2 configs to update (by default do not update)")
	_ = fs.BoolP("no-deploy", "n", false, "only update Z2 configs")
	_ = fs.Bool("parallel", false, "deploy Z2 configs in parallel")
	_ = fs.IntP("z2-retries", "", 3, "Z2 API retries")
	_ = fs.String("packages", "", "packages to set in Z2 configs")

	for _, req := range reqs {
		_ = comm.MarkFlagRequired(req)
	}
}

// ==========================================================================================

func platformTag(str string) string {
	str = strings.ToUpper(str)
	if strings.Contains(str, "TRUSTY") || strings.Contains(str, "14.04") {
		return "LINUX_TRUSTY"
	}
	if strings.Contains(str, "XENIAL") || strings.Contains(str, "16.04") {
		return "LINUX_XENIAL"
	}
	if strings.Contains(str, "BIONIC") || strings.Contains(str, "18.04") {
		return "LINUX_BIONIC"
	}
	if strings.Contains(str, "FOCAL") || strings.Contains(str, "20.04") {
		return "LINUX_FOCAL"
	}
	return "LINUX"
}

func normalizeBranch(branch string) string {
	branch = strings.Trim(branch, "/")
	if branch == "" {
		branch = "trunk"
	} else if branch != "trunk" && !strings.HasPrefix(branch, "branches/") {
		if strings.IndexByte(branch, '/') < 0 {
			branch = "branches/solomon/" + branch
		} else {
			branch = "branches/" + branch
		}
	}
	return branch
}

func normalizePackagePaths(pkgPaths []string) []string {
	for idx, pp := range pkgPaths {
		if strings.IndexByte(pp, '/') < 0 {
			pkgPaths[idx] = "solomon/packages/" + pp + "/pkg.json"
		}
	}
	sort.Strings(pkgPaths)
	return pkgPaths
}

func packagesFromTask(taskID int, sandboxClient *sandbox.Client) ([]string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	task, err := sandboxClient.GetTaskResources(ctx, taskID)
	if err != nil {
		return nil, err
	}
	return task.Packages, nil
}

func splitPP(pp *sandbox.PackageParameters, tasksMax, pkgsPerTaskMax int) []*sandbox.PackageParameters {
	pkgsLen := len(pp.PkgPaths)

	tasks := pkgsLen / pkgsPerTaskMax
	if pkgsLen%pkgsPerTaskMax > 0 {
		tasks++
	}
	if tasks > tasksMax {
		tasks = tasksMax
	}

	ppIdx := 0
	pkgPerTaskFloor := pkgsLen / tasks
	overFloorCount := pkgsLen % tasks
	ppList := make([]*sandbox.PackageParameters, tasks)

	for i := 0; i < tasks; i++ {
		ppList[i] = &sandbox.PackageParameters{}
		*ppList[i] = *pp
		ppIdxNew := ppIdx + pkgPerTaskFloor
		if overFloorCount > 0 {
			ppIdxNew++
			overFloorCount--
		}
		ppList[i].PkgPaths = pp.PkgPaths[ppIdx:ppIdxNew]
		ppIdx = ppIdxNew
	}

	return ppList
}

// ==========================================================================================

func buildPackages(pp *sandbox.PackageParameters, tasksMax, pkgsPerTaskMax int) ([]string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return nil, err
	}
	pp.Platform = platformTag(pp.Platform)
	pp.Branch = normalizeBranch(pp.Branch)
	if pp.Revision == "latest" {
		rev, err := svn.GetLatestRevision(ctx, pp.Branch, "")
		if err != nil {
			return nil, err
		}
		pp.Revision = strconv.Itoa(rev)
	} else {
		pp.Revision = strings.TrimPrefix(pp.Revision, "r")
	}
	pp.PkgPaths = normalizePackagePaths(pp.PkgPaths)

	var wg sync.WaitGroup
	var m sync.Mutex
	result := []string{}
	for _, pp := range splitPP(pp, tasksMax, pkgsPerTaskMax) {
		wg.Add(1)
		go func(pp *sandbox.PackageParameters) {
			pkgs, erx := buildPackagesSingleTask(pp, token)
			m.Lock()
			result = append(result, pkgs...)
			if erx != nil {
				if err != nil {
					err = fmt.Errorf("%v; pkgs=%v: %v", err, pp.PkgPaths, erx)
				} else {
					err = fmt.Errorf("pkgs=%v: %v", pp.PkgPaths, erx)
				}
			}
			m.Unlock()
			wg.Done()
		}(pp)
	}
	wg.Wait()
	log.Printf("All sandbox tasks exited")

	return result, err
}

func buildPackagesSingleTask(pp *sandbox.PackageParameters, token string) ([]string, error) {
	log.Printf("Creating sandbox task: %#v", pp)
	ctx, cancel := context.WithTimeout(context.Background(), 40*time.Second)
	defer cancel()

	sandboxClient := sandbox.NewClient(token)
	taskID, err := sandboxClient.CreateYaPackage(ctx, pp)
	if err != nil {
		log.Printf("Task has failed at CREATE (id=%d): %v", taskID, err)
		return nil, err
	}
	taskURL := "https://sandbox.yandex-team.ru/task/" + strconv.Itoa(taskID) + "/view"

	log.Printf("Starting task id=%d (%s)", taskID, taskURL)
	if err = sandboxClient.StartTask(ctx, taskID); err != nil {
		log.Printf("Task has failed at START (id=%d): %v", taskID, err)
		return nil, err
	}

	prevStatus := sandbox.Unknown
	for {
		ctx, cancel := context.WithTimeout(context.Background(), 25*time.Second)
		status, err := sandboxClient.GetTaskStatus(ctx, taskID)
		cancel()
		if err != nil {
			log.Printf("Error while getting task status (id=%d): %v", taskID, err)
		}
		if _, ok := sandbox.FinishStatusSet[status]; ok {
			log.Printf("Task is done (id=%d): %s", taskID, sandbox.TaskStatusNames[status])
			if status != sandbox.Success {
				return nil, fmt.Errorf("task id=%d failed", taskID)
			}
			break
		} else if status != prevStatus {
			log.Printf("Task status (id=%d): %s, waiting for it to change...", taskID, sandbox.TaskStatusNames[status])
		}
		prevStatus = status
		time.Sleep(10 * time.Second)
	}
	log.Printf("Getting resource names from task id=%d (%s)", taskID, taskURL)
	pkgs, err := packagesFromTask(taskID, sandboxClient)
	if err != nil {
		return nil, err
	}
	log.Printf("Built packages (task id=%d): %v", taskID, pkgs)

	return pkgs, nil
}

// ==========================================================================================

func loadZ2APIKeys() (z2.APIKeys, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return nil, fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return nil, fmt.Errorf("cannot load Z2 API keys: %w", err)
	}
	return apiKeys, nil
}

func deployZ2(cfgList []string, pkgs []string, doDeploy, parallel bool, retries int) error {
	apiKeys, err := loadZ2APIKeys()
	if err != nil {
		return err
	}

	aptPkgs, err := apt.NewPackageList(pkgs...)
	if err != nil {
		return err
	}
	zUpdate := make([]*z2.Worker, 0, len(cfgList))
	for _, cfg := range cfgList {
		zw := z2.NewZW(apiKeys, cfg)
		if n, err := zw.SelectiveEdit(aptPkgs); err != nil {
			return err
		} else if n == 0 {
			log.Printf("No packages found to update in %s, skipping it", cfg)
		} else {
			log.Printf("Updated %d package(s) in %s (%s)", n, cfg, zw.RefURL())
			zUpdate = append(zUpdate, zw)
		}
	}
	if !doDeploy || len(zUpdate) == 0 {
		return nil
	}

	zFailed := make([]*z2.Worker, 0, len(cfgList))
	if parallel {
		zWait := make([]*z2.Worker, 0, len(cfgList))
		for i := 0; i <= retries; i++ {
			for _, zw := range zUpdate {
				log.Printf("Starting to update %s (%s) attempt #%d", zw.Cfg, zw.RefURL(), i+1)
				if err := zw.UpdateStart(); err != nil {
					log.Printf("Update of %s failed: cannot start, %v", zw.Cfg, err)
					zFailed = append(zFailed, zw)
				} else {
					log.Printf("Update of %s started", zw.Cfg)
					zWait = append(zWait, zw)
				}
			}
			for _, zw := range zWait {
				if err := zw.UpdateWait(); err != nil {
					log.Printf("Update of %s failed: %v", zw.Cfg, err)
					zFailed = append(zFailed, zw)
				} else {
					log.Printf("Update of %s succeeded", zw.Cfg)
				}
			}
			if len(zFailed) == 0 {
				break
			}
			if i != retries {
				zUpdate, zFailed, zWait = zFailed, zUpdate[:0], zWait[:0]
				time.Sleep(5 * time.Second)
			}
		}
	} else {
		for _, zw := range zUpdate {
			for i := 0; i <= retries; i++ {
				log.Printf("Deploying to %s (%s), attempt #%d", zw.Cfg, zw.RefURL(), i+1)
				if err := zw.UpdateStart(); err != nil {
					log.Printf("Update of %s failed: cannot start, %v", zw.Cfg, err)
				} else {
					if err := zw.UpdateWait(); err != nil {
						log.Printf("Update of %s failed: %v", zw.Cfg, err)
					} else {
						log.Printf("Update of %s succeeded", zw.Cfg)
						break
					}
				}
				if i == retries {
					zFailed = append(zFailed, zw)
				} else {
					time.Sleep(5 * time.Second)
				}
			}
		}
	}
	if len(zFailed) == 0 {
		return nil
	}
	return fmt.Errorf("%d z2 config failed to deploy", len(zFailed))
}

// ==========================================================================================

func getPackageParametersFromFlags(cmd *cobra.Command) (pp *sandbox.PackageParameters, tasksMax, pkgsPerTaskMax int, err error) {
	var pathStr, repoStr string
	pp = &sandbox.PackageParameters{}

	if pp.Branch, err = cmd.Flags().GetString("branch"); err != nil {
		return
	}
	if pp.Revision, err = cmd.Flags().GetString("revision"); err != nil {
		return
	}
	if pathStr, err = cmd.Flags().GetString("pkg-paths"); err != nil {
		return
	} else if pathStr != "" {
		pp.PkgPaths = brexp.ExpandWithSpaces(strings.Trim(pathStr, " "))
	}
	if pp.Changelog, err = cmd.Flags().GetString("changelog"); err != nil {
		return
	}
	if pp.Strip, err = cmd.Flags().GetBool("strip"); err != nil {
		return
	}
	if pp.FullStrip, err = cmd.Flags().GetBool("full-strip"); err != nil {
		return
	}
	if repoStr, err = cmd.Flags().GetString("repos"); err != nil {
		return
	} else if repoStr != "" {
		pp.Repos = brexp.ExpandWithSpaces(strings.Trim(repoStr, " "))
	}
	if pp.DebianDistribution, err = cmd.Flags().GetString("distribution"); err != nil {
		return
	}
	if pp.DuploadMaxAttempts, err = cmd.Flags().GetInt("dupload-attempts"); err != nil {
		return
	}
	if pp.Platform, err = cmd.Flags().GetString("platform"); err != nil {
		return
	}
	if pp.MemoryGb, err = cmd.Flags().GetInt("memory"); err != nil {
		return
	}
	if pp.DiskGb, err = cmd.Flags().GetInt("disk"); err != nil {
		return
	}
	if pp.Timeout, err = cmd.Flags().GetDuration("timeout"); err != nil {
		return
	}
	if tasksMax, err = cmd.Flags().GetInt("max-tasks"); err != nil {
		return
	}
	if pkgsPerTaskMax, err = cmd.Flags().GetInt("max-pkgs"); err != nil {
		return
	}
	if pp.YtStoreToken, err = cmd.Flags().GetString("yt-token"); err != nil {
		return
	}
	return
}

func getDeployParametersFromFlags(cmd *cobra.Command) (cfgList, pkgs []string, noDeploy, parallel bool, retries int, err error) {
	var cfgStr, pkgString string

	if cfgStr, err = cmd.Flags().GetString("z2-configs"); err != nil {
		return
	} else if cfgStr != "" {
		cfgList = brexp.ExpandWithSpaces(strings.Trim(cfgStr, " "))
	}
	if pkgString, err = cmd.Flags().GetString("packages"); err != nil {
		return
	} else if pkgString != "" {
		pkgs = brexp.ExpandWithSpaces(strings.Trim(pkgString, " "))
	}
	if noDeploy, err = cmd.Flags().GetBool("no-deploy"); err != nil {
		return
	}
	if parallel, err = cmd.Flags().GetBool("parallel"); err != nil {
		return
	}
	if retries, err = cmd.Flags().GetInt("z2-retries"); err != nil {
		return
	}
	return
}

func runBuildPackageCmd(cmd *cobra.Command, args []string) error {
	pp, tasksMax, pkgsPerTaskMax, err := getPackageParametersFromFlags(cmd)
	if err != nil {
		return err
	}
	_, err = buildPackages(pp, tasksMax, pkgsPerTaskMax)
	return err
}

func runDeployPackageCmd(cmd *cobra.Command, args []string) error {
	cfgList, pkgs, noDeploy, parallel, retries, err := getDeployParametersFromFlags(cmd)
	if err != nil {
		return err
	}

	return deployZ2(cfgList, pkgs, !noDeploy, parallel, retries)
}

func runBuildAndDeployPackageCmd(cmd *cobra.Command, args []string) error {
	pp, tasksMax, pkgsPerTaskMax, err := getPackageParametersFromFlags(cmd)
	if err != nil {
		return err
	}
	cfgList, pkgs, noDeploy, parallel, retries, err := getDeployParametersFromFlags(cmd)
	if err != nil {
		return err
	}

	bPkgs, err := buildPackages(pp, tasksMax, pkgsPerTaskMax)
	if err != nil {
		return err
	}
	pkgs = append(pkgs, bPkgs...)
	return deployZ2(cfgList, pkgs, !noDeploy, parallel, retries)
}
