package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	hostsCommon "a.yandex-team.ru/solomon/libs/go/hosts"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/libs/go/utils"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/staffonly"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"net/http"
	"regexp"
	"strconv"
	"time"
)

var alertingClusters = hosts.ParseClusterListProtoText(`
	test {
		replicas { z2_config_id: "SOLOMON_TEST_ALERT", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	pre {
		replicas { z2_config_id: "SOLOMON_PRE_ALERT", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	prod {
		replicas {
			z2_config_id: "SOLOMON_PROD_ALERTING",
			host_pattern: "(sas|vla|myt|man)-\\d{2}",
			dc: "sas,vla,myt,man",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='alerting', cluster='production'", cooldown_minutes: 15 },
				alerts { id: "single-cluster-versions", label_selectors: "service='alerting'", cooldown_minutes: 15 },
				alerts { id: "8ce83daa-78e7-4839-9688-438799c134e4", label_selectors: "endpoint='/api/v2/projects/:projectId/alerts/*', code='503', method='GET'", cooldown_minutes: 15 },
				alerts { id: "alerting-cluster-membership", cooldown_minutes: 15 }
			}
		}
	}
	cloud_pre {
		replicas { z2_config_id: "SOLOMON_CLOUD_PREPROD_ALERTING", host_pattern: "\\d{2}", dc: "sas,vla,myt" }
	}
	cloud_prod {
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_ALERTING",
			host_pattern: "\\d{2}",
			dc: "sas,vla,myt",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "solomon-uptime", label_selectors: "service='alerting', cluster='production'", cooldown_minutes: 15 },
				alerts { id: "single-cluster-versions", label_selectors: "service='alerting'", cooldown_minutes: 15 },
				alerts { id: "8ce83daa-78e7-4839-9688-438799c134e4", label_selectors: "endpoint='/api/v2/projects/:projectId/alerts/*', code='503', method='GET'", cooldown_minutes: 15 }
				alerts { id: "alerting-cluster-membership", cooldown_minutes: 15 }
			}
		}
	}
	cloud_gpn {
		replicas {
			z2_config_id: "SOLOMON_GPN_PROD_ALERTING",
			host_pattern: "\\d{2}",
			dc: "gpn_a,gpn_b"
		}
	}
`)

var commonAlertingPackages = apt.NewPackageListMust(
	"yandex-solomon-alerting")
var internalOrCloudAlertingPackages = apt.NewPackageListMust(
	"yandex-solomon-common-conf",
	"yandex-solomon-alerting-conf")
var gpnAlertingPackages = apt.NewPackageListMust(
	"gpn-solomon-common-conf",
	"gpn-solomon-alerting-conf")

var AlertingCmd = &cobra.Command{
	Use:   fmt.Sprintf("alerting {%s} version [PATTERN]", alertingClusters.AvailableEnvsStr()),
	Short: "Release Alerting",
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) < 2 || len(args) > 3 {
			return fmt.Errorf("Usage: " + cmd.Use)
		}
		return nil
	},
	RunE: runAlertingRecipeCmd,
}

const alertingManagementPort = 4530

func findAlertingLeader(ctx context.Context, env hosts.Env, token string, addresses []hostsCommon.Address) (string, error) {
	staffonlyClient := staffonly.NewClient(env, token, alertingManagementPort, false)
	locationRegexp := regexp.MustCompile("/staffOnly/([a-z0-9.-]+):" + strconv.Itoa(alertingManagementPort) + "/balancer")

	for _, addr := range addresses {
		resp, err := staffonlyClient.Get(ctx, addr.Name, "/balancer")
		if err != nil {
			log.Printf("cannot get response from %s: %v", addr.Name, err)
			log.Println("will try another address after 1 second")
			utils.Sleep(time.Second)
			continue
		}

		// response body is not interesting for this task
		_ = resp.Body.Close()

		switch resp.StatusCode {
		case http.StatusOK:
			// got response from the leader, so current addr is leader
			return addr.Name, nil

		case http.StatusMovedPermanently, http.StatusFound:
			// got response from the follower, which sent redirect to the leader
			location := resp.Header.Get("location")
			leaderHost := locationRegexp.FindStringSubmatch(location)
			if leaderHost == nil || len(leaderHost) != 2 {
				return "", fmt.Errorf("got unexpected location in redirection: '%s'", location)
			}

			return leaderHost[1], nil
		default:
			// got invalid response, so retry later with another address
			log.Printf("got unexpected response status '%s' from '%s'", resp.Status, addr.Name)
			log.Println("will try another address after 1 second")
			utils.Sleep(time.Second)
		}
	}

	return "", fmt.Errorf("cannot find alerting leader host")
}

func runAlertingRecipeCmd(cmd *cobra.Command, args []string) error {
	env := hosts.EnvFromStr(args[0])
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment type: %s", args[0])
	}

	alertingPackages := commonAlertingPackages

	if env.IsGpn() {
		alertingPackages = append(alertingPackages, gpnAlertingPackages...)
	} else {
		alertingPackages = append(alertingPackages, internalOrCloudAlertingPackages...)
	}

	version := args[1]
	if err := alertingPackages.SetVersion(version); err != nil {
		return err
	}

	cluster, err := alertingClusters.FindCluster(env)
	if err != nil {
		return err
	}

	var pattern string
	if len(args) > 2 {
		pattern = args[2]
	}

	replica, err := cluster.FindReplicaConfig(pattern)
	if err != nil {
		return err
	}

	ctx := context.Background()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot load Z2 API keys: %w", err)
	}

	z2Client := z2.NewClient(apiKeys, env.IsCloud())
	hostnames, err := replica.ResolveHosts(ctx, z2Client, pattern)
	if err != nil {
		return err
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	if !env.IsGpn() {
		leaderHost, err := findAlertingLeader(ctx, env, token, addresses)
		if err != nil {
			return err
		}

		if len(addresses) == 1 {
			if addresses[0].Name == leaderHost {
				if cli.Confirm("you are trying to update leader host %s, are you sure: [yes/No]", leaderHost) {
					log.Println("OK, hope you know what you're doing")
				} else {
					log.Println("update terminated")
					return nil
				}
			}
		} else {
			leaderIdx := hostsCommon.FindByName(addresses, leaderHost)
			if leaderIdx == -1 {
				return fmt.Errorf("unknown leader host: %s", leaderHost)
			}
			// leader should be updated last
			hostsCommon.MoveToTheEnd(addresses, leaderIdx)
		}
	}

	eventMaker := &InfraEventMaker{
		Env:            env,
		ServiceName:    "Alerting",
		UpdateDuration: time.Hour,
		Dcs:            replica.Dcs,
	}

	event, err := eventMaker.UpdateOrMakeNewOne(ctx, token)
	if err != nil {
		return err
	}

	var muteMaker *MuteMaker = nil
	if replica.Mutes != nil {
		muteMaker = &MuteMaker{
			Env:            eventMaker.Env,
			ServiceName:    eventMaker.ServiceName,
			UpdateDuration: eventMaker.UpdateDuration,
			Dcs:            eventMaker.Dcs,
			Mutes:          replica.Mutes,
		}
	}

	if muteMaker != nil {
		log.Println(color.BoldYellow("[*] SET MUTES"))
		err = muteMaker.UpdateOrMakeNew(ctx, token)
		if err != nil {
			return err
		}
	}

	sshClient := ussh.NewClusterClient(addresses, env.IsCloud(), "logs")
	defer sshClient.Close()

	log.Println(color.BoldYellow("[*] DOWNLOAD NEW PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr new "+
		"&& mkdir new && cd new "+
		"&& sudo apt-get update "+
		"&& apt-get download "+alertingPackages.String())

	log.Println(color.BoldYellow("[*] DOWNLOAD OLD PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr old "+
		"&& mkdir old && cd old "+
		"&& dpkg-query --showformat='${Package}=${Version}\\n' --show "+alertingPackages.NamesString()+
		" | grep stable | xargs apt-get download")

	log.Println(color.BoldYellow("[*] INSTALL NEW PACKAGES"))
	var result *ussh.ClientRunResult
	if env.IsCloud() {
		result = sshClient.RunSequentially("sudo dpkg -i -E new/*.deb && rm -fr new", time.Minute*5)
	} else {
		// Commented before https://st.yandex-team.ru/SOLOMON-8626 gets fixed
		// stop := func(address string) error {
		// 	return changeState(ctx, env, token, addresses, address, false)
		// }
		// start := func(address string) error {
		// 	return changeState(ctx, env, token, addresses, address, true)
		// }
		// result = sshClient.RunParallelWithWrapCall(4, "sudo dpkg -i -E new/*.deb && rm -fr new", stop, start, time.Minute*1)
	
		// 15+15+6 = 36 hosts, 2 min delay - approx 1.5h deploy
		result = sshClient.RunSequentially("sudo dpkg -i -E new/*.deb && rm -fr new", time.Minute*2)
	}

	if len(hostnames) > 1 {
		if cli.CanContinueUpdate(result.FailedCount) {
			log.Println(color.BoldYellow("[*] UPDATE Z2 CONFIG"))
			if err = updateZ2Config(ctx, z2Client, replica.Z2ConfigID, alertingPackages); err != nil {
				return err
			}
		} else {
			log.Println(color.BoldRed("[*] Z2 CONFIG WASN'T UPDATE"))
		}

		log.Println(color.BoldYellow("[*] CLOSE INFRA EVENT"))
		if err = eventMaker.FinishEvent(ctx, token, event); err != nil {
			return err
		}

		if muteMaker != nil {
			log.Println(color.BoldYellow("[*] ADJUST MUTES FINISH TIME"))
			if err = muteMaker.AdjustFinishTime(ctx, token); err != nil {
				return err
			}
		}
	}

	return nil
}

func changeState(ctx context.Context, env hosts.Env, token string, addresses []hostsCommon.Address, addrName string, flag bool) error {
	leaderHost, err := findAlertingLeader(ctx, env, token, addresses)
	if err != nil {
		return err
	}
	locationRegexp := regexp.MustCompile("([a-z0-9.-]+):" + strconv.Itoa(alertingManagementPort) + "/balancer")
	staffonlyClient := staffonly.NewClient(env, token, alertingManagementPort, false)
	URL := "/balancer/nodeActive?node=" + addrName + "&flag=" + strconv.FormatBool(flag)
	resp, err := staffonlyClient.Get(ctx, leaderHost, URL)
	if err != nil {
		return fmt.Errorf("cannot get response from %s: %v", addrName, err)
	}

	_ = resp.Body.Close()

	switch resp.StatusCode {
	case http.StatusOK:
		return nil
	case http.StatusMovedPermanently, http.StatusFound:
		// got response from the follower, which sent redirect to the leader
		location := resp.Header.Get("location")
		leaderHost := locationRegexp.FindStringSubmatch(location)
		if leaderHost == nil {
			return fmt.Errorf("cannot change alerting node state to %s", strconv.FormatBool(flag))
		}

		return nil
	}
	return fmt.Errorf("cannot change alerting node state to %s", strconv.FormatBool(flag))
}
