package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"time"
)

var backupClusters = hosts.ParseClusterListProtoText(`
	prod {
		replicas {
			z2_config_id: "SOLOMON_PROD_STORAGE_SAS",
			host_pattern: "sas-\\d{3}",
			dc: "sas",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='backup', cluster='storage_sas'", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_PROD_STORAGE_VLA",
			host_pattern: "vla-\\d{3}",
			dc: "vla",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='backup', cluster='storage_vla'", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_PROD_HOST_GATEWAY",
			host_pattern: "(sas|vla|myt)-\\d{2}",
			dc: "sas,vla,myt",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='backup', cluster='kfront'", cooldown_minutes: 10 }
			}
		}
	}
	cloud_prod {
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_STOCKPILE_SAS",
			host_pattern: "sas-\\d{2}",
			dc: "sas",
			mutes {
				project_id: "solomon_cloud"
				alerts { id: "solomon-uptime", label_selectors: "service='backup', cluster='storage_sas'", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_STOCKPILE_VLA",
			host_pattern: "vla-\\d{2}",
			dc: "vla",
			mutes {
				project_id: "solomon_cloud"
				alerts { id: "solomon-uptime", label_selectors: "service='backup', cluster='storage_vla'", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_GATEWAY",
			host_pattern: "\\d{2}",
			dc: "sas,vla,myt",
			mutes {
				project_id: "solomon_cloud"
				alerts { id: "solomon-uptime", label_selectors: "service='backup', cluster='production'", cooldown_minutes: 10 }
			}
		}
	}
`)

var backupPackages = apt.NewPackageListMust("yandex-solomon-backup")

var BackupCmd = &cobra.Command{
	Use:   fmt.Sprintf("backup {%s} version [PATTERN]", backupClusters.AvailableEnvsStr()),
	Short: "Release Backup Tool",
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) < 2 || len(args) > 3 {
			return fmt.Errorf("Usage: " + cmd.Use)
		}
		return nil
	},
	RunE: runBackupRecipeCmd,
}

func runBackupRecipeCmd(cmd *cobra.Command, args []string) error {
	env := hosts.EnvFromStr(args[0])
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment type: %s", args[0])
	}

	version := args[1]
	if err := backupPackages.SetVersion(version); err != nil {
		return err
	}

	cluster, err := backupClusters.FindCluster(env)
	if err != nil {
		return err
	}

	var pattern string
	if len(args) > 2 {
		pattern = args[2]
	}

	replica, err := cluster.FindReplicaConfig(pattern)
	if err != nil {
		return err
	}

	ctx := context.Background()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot load Z2 API keys: %w", err)
	}

	z2Client := z2.NewClient(apiKeys, env.IsCloud())
	hostnames, err := replica.ResolveHosts(ctx, z2Client, pattern)
	if err != nil {
		return err
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	eventMaker := &InfraEventMaker{
		Env:            env,
		ServiceName:    "Backup Tool",
		UpdateDuration: 20 * time.Minute,
		Dcs:            replica.Dcs,
	}

	event, err := eventMaker.UpdateOrMakeNewOne(ctx, token)
	if err != nil {
		return err
	}

	var muteMaker *MuteMaker = nil
	if replica.Mutes != nil {
		muteMaker = &MuteMaker{
			Env:            eventMaker.Env,
			ServiceName:    eventMaker.ServiceName,
			UpdateDuration: eventMaker.UpdateDuration,
			Dcs:            eventMaker.Dcs,
			Mutes:          replica.Mutes,
		}
	}

	if muteMaker != nil {
		log.Println(color.BoldYellow("[*] SET MUTES"))
		err = muteMaker.UpdateOrMakeNew(ctx, token)
		if err != nil {
			return err
		}
	}

	sshClient := ussh.NewClusterClient(addresses, env.IsCloud(), "logs")
	defer sshClient.Close()

	log.Println(color.BoldYellow("[*] DOWNLOAD NEW PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr new "+
		"&& mkdir new && cd new "+
		"&& sudo apt-get update "+
		"&& apt-get download "+backupPackages.String())

	log.Println(color.BoldYellow("[*] DOWNLOAD OLD PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr old "+
		"&& mkdir old && cd old "+
		"&& dpkg-query --showformat='${Package}=${Version}\\n' --show "+backupPackages.NamesString()+" | grep stable | xargs apt-get download")

	log.Println(color.BoldYellow("[*] INSTALL NEW PACKAGES"))
	result := sshClient.RunParallel(8, "sudo dpkg -i -E new/*.deb && rm -fr new")

	if len(hostnames) > 1 {
		if cli.CanContinueUpdate(result.FailedCount) {
			log.Println(color.BoldYellow("[*] UPDATE Z2 CONFIG"))
			if err = updateZ2Config(ctx, z2Client, replica.Z2ConfigID, backupPackages); err != nil {
				return err
			}
		} else {
			log.Println(color.BoldRed("[*] Z2 CONFIG WASN'T UPDATE"))
		}

		log.Println(color.BoldYellow("[*] CLOSE INFRA EVENT"))
		if err = eventMaker.FinishEvent(ctx, token, event); err != nil {
			return err
		}

		if muteMaker != nil {
			log.Println(color.BoldYellow("[*] ADJUST MUTES FINISH TIME"))
			if err = muteMaker.AdjustFinishTime(ctx, token); err != nil {
				return err
			}
		}
	}

	return nil
}
