package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/tracker"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/infra"
	"a.yandex-team.ru/solomon/tools/release/internal/mutes"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"log"
	"strings"
	"time"
)

const (
	maxParallelism = 50
)

type InfraEventMaker struct {
	Env            hosts.Env
	ServiceName    string
	UpdateDuration time.Duration
	Dcs            []hosts.Dc
}

type MuteSpecification struct {
	AlertIDPattern string
	LabelSelectors string
	ExtraCoolDown  time.Duration
}

type MuteMaker struct {
	Env            hosts.Env
	ServiceName    string
	UpdateDuration time.Duration
	Dcs            []hosts.Dc
	Mutes          *hosts.MuteList
}

const (
	javaIsoTimeFormat = "2006-01-02T15:04:05.999Z"
)

func findReleaseIssue(ctx context.Context, token string) (*tracker.Issue, error) {
	trackerClient := tracker.NewClient(token)
	issues, err := trackerClient.FindIssues(ctx, tracker.Filter{
		Queue:      &tracker.SolomonQueue,
		Components: []int{tracker.SolomonReleaseID},
		Statuses:   []string{tracker.StatusNew, tracker.StatusOpen, tracker.StatusInProgress},
	})
	if err != nil {
		return nil, fmt.Errorf("cannot find open release issues: %w", err)
	}

	if len(issues) == 0 {
		return nil, fmt.Errorf("there is no release issue yet, try to create a new one before release")
	}

	if len(issues) == 1 {
		return &issues[0], nil
	}

	keys := make([]string, 0, len(issues))
	for _, issue := range issues {
		keys = append(keys, *issue.Key)
	}
	return nil, fmt.Errorf("too many open/in progress release issues, "+
		"try to close them except the one: %s", strings.Join(keys, ", "))
}

func (m *InfraEventMaker) UpdateOrMakeNewOne(ctx context.Context, token string) (*infra.Event, error) {
	ctx, cancel := context.WithTimeout(ctx, time.Second*10)
	defer cancel()

	serviceEnv, err := infra.FindServiceEnv("solomon", m.Env)
	if err != nil {
		return nil, fmt.Errorf("cannot find service environment: %w", err)
	}

	infraClient := infra.NewClient(token)
	event, err := infraClient.FindCurrentEvents(ctx, *serviceEnv, m.ServiceName)
	if err != nil {
		return nil, fmt.Errorf("cannot find infra events: %w", err)
	}

	if event != nil {
		log.Println("found infra event", event.URL(), event.Title)

		finishTime := time.Now().Add(m.UpdateDuration)
		event.FinishTime = finishTime.Unix()

		if err = infraClient.UpdateEvent(ctx, event); err != nil {
			return nil, fmt.Errorf("cannot update infra event: %w", err)
		}

		log.Println("event finish time updated up to", finishTime.Format("15:04"))
		return event, nil
	}

	issue, err := findReleaseIssue(ctx, token)
	if err != nil {
		return nil, err
	}

	event = serviceEnv.NewEvent(
		fmt.Sprintf("Релиз %s", m.ServiceName),
		fmt.Sprintf("Плановый релиз новой версии %s, без downtime", m.ServiceName),
		m.UpdateDuration,
		*issue.Key,
		m.Dcs)

	if err = infraClient.CreateEvent(ctx, event); err != nil {
		return nil, fmt.Errorf("cannot create infra event: %w", err)
	}

	log.Println("created new infra event", event.URL())
	return event, nil
}

func (m *InfraEventMaker) FinishEvent(ctx context.Context, token string, event *infra.Event) error {
	event.FinishTime = time.Now().Unix()

	infraClient := infra.NewClient(token)
	if err := infraClient.UpdateEvent(ctx, event); err != nil {
		return fmt.Errorf("cannot finish infra event: %w", err)
	}

	log.Println("infra event", event.URL(), "finished")
	return nil
}

func updateZ2Config(ctx context.Context, client z2.Client, configID string, packages apt.PackageList) error {
	if err := client.EditItems(ctx, configID, packages); err != nil {
		return fmt.Errorf("cannot edit Z2 config %s: %w", configID, err)
	}

	// TODO: check that config has only our changes
	if err := client.Update(ctx, configID); err != nil {
		return fmt.Errorf("cannot update Z2 config %s: %w", configID, err)
	}

	log.Printf("go to %s to see update status\n", client.ControlPanelURL(configID))

	for i := 0; ; i++ {
		status, err := client.UpdateStatus(context.Background(), configID)
		if err != nil {
			return fmt.Errorf("cannot get update status in %s: %w", configID, err)
		}

		fmt.Printf("\033[100DUpdate status %s (%d)", status.UpdateStatus, i)
		if status.IsFinished() {
			fmt.Print("\033[100D")
			status.Print(configID)
			break
		}

		time.Sleep(2 * time.Second)
	}

	return nil
}

func (m *MuteMaker) makeDescription() string {
	dcs := make([]string, len(m.Dcs))
	for i, dc := range m.Dcs {
		dcs[i] = dc.String()
	}
	return fmt.Sprintf("Релиз %s на %s в %s", m.ServiceName, m.Env.String(), strings.Join(dcs, ","))
}

func (m *MuteMaker) makeMuteBlueprints(ticketID string) []mutes.Mute {
	description := m.makeDescription()

	dts := make([]mutes.Mute, len(m.Mutes.Alerts))
	for i, alert := range m.Mutes.Alerts {
		finishTime := time.Now().Add(m.UpdateDuration).Add(alert.CoolDown).UTC()
		labelSelectors := alert.LabelSelectors
		if !strings.HasPrefix(labelSelectors, "{") {
			labelSelectors = "{" + labelSelectors + "}"
		}
		var projectID string
		if len(alert.ProjectID) > 0 {
			projectID = alert.ProjectID
		} else {
			projectID = m.Mutes.ProjectID
		}
		dts[i] = mutes.Mute{
			ProjectID:   projectID,
			Description: description,
			Name:        fmt.Sprintf("Релиз %s", m.ServiceName),
			TicketID:    ticketID,
			To:          finishTime.Format(javaIsoTimeFormat),
			Type: mutes.MuteType{
				Selectors: &mutes.SelectorsMute{
					AlertSelector:  fmt.Sprintf("alert='%s'", alert.ID),
					LabelSelectors: labelSelectors,
				},
			},
		}
	}
	return dts
}

func (m *MuteMaker) UpdateOrMakeNew(ctx context.Context, token string) error {
	ctx, cancel := context.WithTimeout(ctx, time.Second*10)
	defer cancel()

	if m.Mutes == nil {
		log.Println("Mutes not configured")
		return nil
	}

	issue, err := findReleaseIssue(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot find release ticket: %w", err)
	}

	dts := m.makeMuteBlueprints(*issue.Key)

	mutesClient := mutes.NewClient(token)
	err = mutesClient.FillExistingMutesID(ctx, dts)

	if err != nil {
		return fmt.Errorf("cannot fill ids for current mutes: %w", err)
	}

	for _, mute := range dts {
		if mute.ID != nil {
			log.Println("updating mute", mute.URL())
			err = mutesClient.UpdateMute(ctx, &mute)
			if err != nil {
				return fmt.Errorf("failed to update mute %s: %w", mute.URL(), err)
			}
		} else {
			oldSelectors := *mute.Type.Selectors
			err = mutesClient.CreateMute(ctx, &mute)
			if oldSelectors != *mute.Type.Selectors {
				log.Printf(color.BoldRed("Warning")+": alert specification is not in canonical form,"+
					" release tool wont be able to update this mute:\n"+
					"Given %v, canonical %v\n", oldSelectors, *mute.Type.Selectors)
			}
			if err != nil {
				return fmt.Errorf("failed to create mute: %w", err)
			}
			log.Println("created mute", mute.URL())
		}
	}

	return nil
}

func (m *MuteMaker) AdjustFinishTime(ctx context.Context, token string) error {
	ctx, cancel := context.WithTimeout(ctx, time.Second*10)
	defer cancel()

	if m.Mutes == nil {
		return nil
	}

	issue, err := findReleaseIssue(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot find release ticket: %w", err)
	}

	dts := m.makeMuteBlueprints(*issue.Key)

	mutesClient := mutes.NewClient(token)
	err = mutesClient.FillExistingMutesID(ctx, dts)

	if err != nil {
		return fmt.Errorf("cannot fill ids for current mutes: %w", err)
	}

	for i, mute := range dts {
		if mute.ID != nil {
			log.Println("Updating mute finish time", mute.URL())
			finishTime := time.Now().Add(m.Mutes.Alerts[i].CoolDown).UTC()
			mute.To = finishTime.Format(javaIsoTimeFormat)
			err = mutesClient.UpdateMute(ctx, &mute)
			if err != nil {
				return fmt.Errorf("failed to update mute %s: %w", mute.URL(), err)
			}
		}
	}

	return nil
}
