package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"time"
)

var dataproxyClusters = hosts.ParseClusterListProtoText(`
	test {
		replicas { z2_config_id: "SOLOMON_TEST_DATAPROXY", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	pre {
		replicas { z2_config_id: "SOLOMON_PRE_DATAPROXY", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	prod {
		replicas {
			z2_config_id: "SOLOMON_PROD_DATAPROXY",
			host_pattern: "(sas|vla|myt|man)-\\d{2}",
			dc: "sas,vla,myt,man",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='dataproxy', cluster='production'", cooldown_minutes: 15 }
			}
		}
	}
`)

var dataproxyPackages = apt.NewPackageListMust(
	"yandex-solomon-dataproxy",
	"yandex-solomon-dataproxy-conf")

var DataproxyCmd = &cobra.Command{
	Use:   fmt.Sprintf("dataproxy {%s} version [PATTERN]", dataproxyClusters.AvailableEnvsStr()),
	Short: "Release Dataproxy",
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) < 2 || len(args) > 3 {
			return fmt.Errorf("Usage: " + cmd.Use)
		}
		return nil
	},
	RunE: runDataproxyRecipeCmd,
}

func runDataproxyRecipeCmd(cmd *cobra.Command, args []string) error {
	env := hosts.EnvFromStr(args[0])
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment type: %s", args[0])
	}

	version := args[1]
	if err := dataproxyPackages.SetVersion(version); err != nil {
		return err
	}

	cluster, err := dataproxyClusters.FindCluster(env)
	if err != nil {
		return err
	}

	var pattern string
	if len(args) > 2 {
		pattern = args[2]
	}

	replica, err := cluster.FindReplicaConfig(pattern)
	if err != nil {
		return err
	}

	ctx := context.Background()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot load Z2 API keys: %w", err)
	}

	z2Client := z2.NewClient(apiKeys, env.IsCloud())
	hostnames, err := replica.ResolveHosts(ctx, z2Client, pattern)
	if err != nil {
		return err
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	eventMaker := &InfraEventMaker{
		Env:            env,
		ServiceName:    "Dataproxy",
		UpdateDuration: time.Minute * 30,
		Dcs:            replica.Dcs,
	}

	event, err := eventMaker.UpdateOrMakeNewOne(ctx, token)
	if err != nil {
		return err
	}

	var muteMaker *MuteMaker = nil
	if replica.Mutes != nil {
		muteMaker = &MuteMaker{
			Env:            eventMaker.Env,
			ServiceName:    eventMaker.ServiceName,
			UpdateDuration: eventMaker.UpdateDuration,
			Dcs:            eventMaker.Dcs,
			Mutes:          replica.Mutes,
		}
	}

	if muteMaker != nil {
		log.Println(color.BoldYellow("[*] SET MUTES"))
		err = muteMaker.UpdateOrMakeNew(ctx, token)
		if err != nil {
			return err
		}
	}

	sshClient := ussh.NewClusterClient(addresses, env.IsCloud(), "logs")
	defer sshClient.Close()

	log.Println(color.BoldYellow("[*] DOWNLOAD NEW PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr new "+
		"&& mkdir new && cd new "+
		"&& sudo apt-get update "+
		"&& apt-get download "+dataproxyPackages.String())

	log.Println(color.BoldYellow("[*] DOWNLOAD OLD PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr old "+
		"&& mkdir old && cd old "+
		"&& dpkg-query --showformat='${Package}=${Version}\\n' --show 'yandex-solomon*' | grep stable | xargs apt-get download")

	log.Println(color.BoldYellow("[*] INSTALL NEW PACKAGES"))
	result := sshClient.RunSequentially("sudo dpkg -i -E new/*.deb && rm -fr new", time.Minute)

	if len(hostnames) > 1 {
		if cli.CanContinueUpdate(result.FailedCount) {
			log.Println(color.BoldYellow("[*] UPDATE Z2 CONFIG"))
			if err = updateZ2Config(ctx, z2Client, replica.Z2ConfigID, dataproxyPackages); err != nil {
				return err
			}
		} else {
			log.Println(color.BoldRed("[*] Z2 CONFIG WASN'T UPDATE"))
		}

		log.Println(color.BoldYellow("[*] CLOSE INFRA EVENT"))
		if err = eventMaker.FinishEvent(ctx, token, event); err != nil {
			return err
		}

		if muteMaker != nil {
			log.Println(color.BoldYellow("[*] ADJUST MUTES FINISH TIME"))
			if err = muteMaker.AdjustFinishTime(ctx, token); err != nil {
				return err
			}
		}
	}

	return nil
}
