package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"time"
)

var fetcherClusters = hosts.ParseClusterListProtoText(`
	test {
		replicas { z2_config_id: "SOLOMON_TEST", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	pre {
		replicas { z2_config_id: "SOLOMON_PRE_FETCHER", host_pattern: "(man-)?\\d{3}", dc: "man" }
	}
	prod {
		replicas {
			z2_config_id: "SOLOMON_PROD_FETCHER_SAS",
			host_pattern: "sas-\\d{3}",
			dc: "sas"
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='fetcher', cluster='production', host='*sas*'", cooldown_minutes: 10 }
				alerts { id: "*-diff-dc-total|coremon-dc-write-diff-vla-sas", cooldown_minutes: 10 }
				alerts { id: "multi-cluster-versions", label_selectors: "service='fetcher', host='Sas'", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_PROD_FETCHER_VLA",
			host_pattern: "vla-\\d{3}",
			dc: "vla"
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='fetcher', cluster='production', host='*vla*'", cooldown_minutes: 10 }
				alerts { id: "*-diff-dc-total|coremon-dc-write-diff-vla-sas", cooldown_minutes: 10 }
				alerts { id: "multi-cluster-versions", label_selectors: "service='fetcher', host='Vla'", cooldown_minutes: 10 }
			}
		}
	}
	cloud_pre {
		replicas { z2_config_id: "SOLOMON_CLOUD_PREPROD_FETCHER", host_pattern: "\\d{2}", dc: "sas" }
	}
	cloud_prod {
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_FETCHER_SAS",
			host_pattern: "sas-\\d{2}",
			dc: "sas",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "solomon-uptime", label_selectors: "service='fetcher', cluster='production', host='*sas*'", cooldown_minutes: 10 }
				alerts { id: "*-diff-dc-total|coremon-dc-write-diff-vla-sas", cooldown_minutes: 10 }
				alerts { id: "multi-cluster-versions", label_selectors: "service='fetcher', host='Sas'", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_FETCHER_VLA",
			host_pattern: "vla-\\d{2}",
			dc: "vla",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "solomon-uptime", label_selectors: "service='fetcher', cluster='production', host='*vla*'", cooldown_minutes: 10 }
				alerts { id: "*-diff-dc-total|coremon-dc-write-diff-vla-sas", cooldown_minutes: 10 }
				alerts { id: "multi-cluster-versions", label_selectors: "service='fetcher', host='Vla'", cooldown_minutes: 10 }
			}
		}
	}
	cloud_gpn {
		replicas {
			z2_config_id: "SOLOMON_GPN_PROD_FETCHER_A",
			host_pattern: "a-\\d{2}",
			dc: "gpn_a"
		}
		replicas {
			z2_config_id: "SOLOMON_GPN_PROD_FETCHER_B",
			host_pattern: "b-\\d{2}",
			dc: "gpn_b"
		}
	}
`)

var commonFetcherPackages = apt.NewPackageListMust(
	"yandex-solomon-fetcher")
var internalOrCloudFetcherPackages = apt.NewPackageListMust(
	"yandex-solomon-fetcher-conf")
var gpnFetcherPackages = apt.NewPackageListMust(
	"gpn-solomon-fetcher-conf")

var FetcherCmd = &cobra.Command{
	Use:   fmt.Sprintf("fetcher {%s} version [PATTERN]", fetcherClusters.AvailableEnvsStr()),
	Short: "Release Fetcher",
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) < 2 || len(args) > 3 {
			return fmt.Errorf("Usage: " + cmd.Use)
		}
		return nil
	},
	RunE: runFetcherRecipeCmd,
}

func runFetcherRecipeCmd(cmd *cobra.Command, args []string) error {
	env := hosts.EnvFromStr(args[0])
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment type: %s", args[0])
	}

	fetcherPackages := commonFetcherPackages

	if env.IsGpn() {
		fetcherPackages = append(fetcherPackages, gpnFetcherPackages...)
	} else {
		fetcherPackages = append(fetcherPackages, internalOrCloudFetcherPackages...)
	}

	version := args[1]
	if err := fetcherPackages.SetVersion(version); err != nil {
		return err
	}

	cluster, err := fetcherClusters.FindCluster(env)
	if err != nil {
		return err
	}

	var pattern string
	if len(args) > 2 {
		pattern = args[2]
	}

	if pattern == "gpn_a" {
		pattern = "SOLOMON_GPN_PROD_FETCHER_A"
	} else if pattern == "gpn_b" {
		pattern = "SOLOMON_GPN_PROD_FETCHER_B"
	}

	replica, err := cluster.FindReplicaConfig(pattern)
	if err != nil {
		return err
	}

	ctx := context.Background()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot load Z2 API keys: %w", err)
	}

	z2Client := z2.NewClient(apiKeys, env.IsCloud())
	hostnames, err := replica.ResolveHosts(ctx, z2Client, pattern)
	if err != nil {
		return err
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	infraServiceName := "Fetcher"
	if env.IsGpn() {
		infraServiceName += " в реплику " + replica.Dcs[0].String()
	}

	eventMaker := &InfraEventMaker{
		Env:            env,
		ServiceName:    infraServiceName,
		UpdateDuration: time.Minute * 30,
		Dcs:            replica.Dcs,
	}

	event, err := eventMaker.UpdateOrMakeNewOne(ctx, token)
	if err != nil {
		return err
	}

	var muteMaker *MuteMaker = nil
	if replica.Mutes != nil {
		muteMaker = &MuteMaker{
			Env:            eventMaker.Env,
			ServiceName:    eventMaker.ServiceName,
			UpdateDuration: eventMaker.UpdateDuration,
			Dcs:            eventMaker.Dcs,
			Mutes:          replica.Mutes,
		}
	}

	if muteMaker != nil {
		log.Println(color.BoldYellow("[*] SET MUTES"))
		err = muteMaker.UpdateOrMakeNew(ctx, token)
		if err != nil {
			return err
		}
	}

	sshClient := ussh.NewClusterClient(addresses, env.IsCloud(), "logs")
	defer sshClient.Close()

	log.Println(color.BoldYellow("[*] DOWNLOAD NEW PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr new "+
		"&& mkdir new && cd new "+
		"&& sudo apt-get update "+
		"&& apt-get download "+fetcherPackages.String())

	log.Println(color.BoldYellow("[*] DOWNLOAD OLD PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr old "+
		"&& mkdir old && cd old "+
		"&& dpkg-query --showformat='${Package}=${Version}\\n' --show "+fetcherPackages.NamesString()+
		" | grep stable | xargs apt-get download")

	log.Println(color.BoldYellow("[*] INSTALL NEW PACKAGES"))
	var result *ussh.ClientRunResult
	if env.IsCloud() {
		result = sshClient.RunSequentially("sudo dpkg -i -E new/*.deb && rm -fr new", time.Minute)
	} else {
		result = sshClient.RunParallel(16, "sudo dpkg -i -E new/*.deb && rm -fr new")
	}

	if len(hostnames) > 1 {
		if cli.CanContinueUpdate(result.FailedCount) {
			log.Println(color.BoldYellow("[*] UPDATE Z2 CONFIG"))
			if err = updateZ2Config(ctx, z2Client, replica.Z2ConfigID, fetcherPackages); err != nil {
				return err
			}
		} else {
			log.Println(color.BoldRed("[*] Z2 CONFIG WASN'T UPDATE"))
		}

		log.Println(color.BoldYellow("[*] CLOSE INFRA EVENT"))
		if err = eventMaker.FinishEvent(ctx, token, event); err != nil {
			return err
		}

		if muteMaker != nil {
			log.Println(color.BoldYellow("[*] ADJUST MUTES FINISH TIME"))
			if err = muteMaker.AdjustFinishTime(ctx, token); err != nil {
				return err
			}
		}
	}

	return nil
}
