package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"time"
)

var projectManagerClusters = hosts.ParseClusterListProtoText(`
	test {
		replicas { z2_config_id: "SOLOMON_TEST_PM", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	pre {
		replicas { z2_config_id: "SOLOMON_PRE_PM", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	prod {
		replicas {
			z2_config_id: "SOLOMON_PROD_PM",
			host_pattern: "(sas|vla|myt|man)-\\d{2}",
			dc: "sas,vla,myt,man"
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='project-manager', cluster='production'", cooldown_minutes: 15 },
				alerts { id: "single-cluster-versions", label_selectors: "service='project-manager'", cooldown_minutes: 15 }
			}
		}
	}
	cloud_pre {
		replicas { z2_config_id: "SOLOMON_CLOUD_PREPROD_PM", host_pattern: "\\d{2}", dc: "sas,vla,myt" }
	}
	cloud_prod {
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_PM",
			host_pattern: "\\d{2}",
			dc: "sas,vla,myt",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "solomon-uptime", label_selectors: "service='project-manager', cluster='production'", cooldown_minutes: 15 },
				alerts { id: "single-cluster-versions", label_selectors: "service='project-manager'", cooldown_minutes: 15 }
			}
		}
	}
	cloud_gpn {
		replicas {
			z2_config_id: "SOLOMON_GPN_PROD_PM",
			host_pattern: "\\d{2}",
			dc: "gpn_a,gpn_b",
		}
	}
`)

var commonProjectManagerPackages = apt.NewPackageListMust(
	"yandex-solomon-project-manager")

var internalOrCloudProjectManagerPackages = apt.NewPackageListMust(
	"yandex-solomon-common-conf",
	"yandex-solomon-project-manager-conf")

var gpnProjectManagerPackages = apt.NewPackageListMust(
	"gpn-solomon-common-conf",
	"gpn-solomon-project-manager-conf")

var ProjectManagerCmd = &cobra.Command{
	Use:   fmt.Sprintf("project_manager {%s} version [PATTERN]", projectManagerClusters.AvailableEnvsStr()),
	Short: "Release ProjectManager",
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) < 2 || len(args) > 3 {
			return fmt.Errorf("Usage: " + cmd.Use)
		}
		return nil
	},
	RunE: runProjectManagerRecipeCmd,
}

func runProjectManagerRecipeCmd(cmd *cobra.Command, args []string) error {
	env := hosts.EnvFromStr(args[0])
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment type: %s", args[0])
	}

	packages := commonProjectManagerPackages

	if env.IsGpn() {
		packages = append(packages, gpnProjectManagerPackages...)
	} else {
		packages = append(packages, internalOrCloudProjectManagerPackages...)
	}

	version := args[1]
	if err := packages.SetVersion(version); err != nil {
		return err
	}

	cluster, err := projectManagerClusters.FindCluster(env)
	if err != nil {
		return err
	}

	var pattern string
	if len(args) > 2 {
		pattern = args[2]
	}

	replica, err := cluster.FindReplicaConfig(pattern)
	if err != nil {
		return err
	}

	ctx := context.Background()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot load Z2 API keys: %w", err)
	}

	z2Client := z2.NewClient(apiKeys, env.IsCloud())
	hostnames, err := replica.ResolveHosts(ctx, z2Client, pattern)
	if err != nil {
		return err
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	eventMaker := &InfraEventMaker{
		Env:            env,
		ServiceName:    "ProjectManager",
		UpdateDuration: time.Hour,
		Dcs:            replica.Dcs,
	}

	event, err := eventMaker.UpdateOrMakeNewOne(ctx, token)
	if err != nil {
		return err
	}

	var muteMaker *MuteMaker = nil
	if replica.Mutes != nil {
		muteMaker = &MuteMaker{
			Env:            eventMaker.Env,
			ServiceName:    eventMaker.ServiceName,
			UpdateDuration: eventMaker.UpdateDuration,
			Dcs:            eventMaker.Dcs,
			Mutes:          replica.Mutes,
		}
	}

	if muteMaker != nil {
		log.Println(color.BoldYellow("[*] SET MUTES"))
		err = muteMaker.UpdateOrMakeNew(ctx, token)
		if err != nil {
			return err
		}
	}

	sshClient := ussh.NewClusterClient(addresses, env.IsCloud(), "logs")
	defer sshClient.Close()

	log.Println(color.BoldYellow("[*] DOWNLOAD NEW PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr new "+
		"&& mkdir new && cd new "+
		"&& sudo apt-get update "+
		"&& apt-get download "+packages.String())

	log.Println(color.BoldYellow("[*] DOWNLOAD OLD PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr old "+
		"&& mkdir old && cd old "+
		"&& dpkg-query --showformat='${Package}=${Version}\\n' --show "+packages.NamesString()+
		" | grep stable | xargs apt-get download")

	log.Println(color.BoldYellow("[*] INSTALL NEW PACKAGES"))
	result := sshClient.RunSequentially("sudo dpkg -i -E new/*.deb && rm -fr new", time.Minute*2)

	if len(hostnames) > 1 {
		if cli.CanContinueUpdate(result.FailedCount) {
			log.Println(color.BoldYellow("[*] UPDATE Z2 CONFIG"))
			if err = updateZ2Config(ctx, z2Client, replica.Z2ConfigID, packages); err != nil {
				return err
			}
		} else {
			log.Println(color.BoldRed("[*] Z2 CONFIG WASN'T UPDATE"))
		}

		log.Println(color.BoldYellow("[*] CLOSE INFRA EVENT"))
		if err = eventMaker.FinishEvent(ctx, token, event); err != nil {
			return err
		}

		if muteMaker != nil {
			log.Println(color.BoldYellow("[*] ADJUST MUTES FINISH TIME"))
			if err = muteMaker.AdjustFinishTime(ctx, token); err != nil {
				return err
			}
		}
	}

	return nil
}
