package recipe

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	hostsCommon "a.yandex-team.ru/solomon/libs/go/hosts"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/libs/go/utils"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/cli"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/staffonly"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"net/http"
	"time"
)

var stockpileClusters = hosts.ParseClusterListProtoText(`
	test {
		replicas { z2_config_id: "SOLOMON_TEST", host_pattern: "(sas-)?\\d{2}", dc: "sas" }
	}
	pre {
		replicas {
			z2_config_id: "SOLOMON_PRE_STOCKPILE",
			host_pattern: "(man-)?\\d{3}",
			dc: "man",
			mutes {
				project_id: "solomon",
				alerts { id: "pre-solomon-uptime", label_selectors: "service='stockpile'", cooldown_minutes: 10 }
			}
		}
	}
	prod {
		replicas {
			z2_config_id: "SOLOMON_PROD_STOCKPILE_SAS",
			host_pattern: "sas-\\d{3}",
			dc: "sas",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='stockpile', cluster='stockpile_sas'", cooldown_minutes: 10 }
				alerts { project_id: "graphite", id: "ed960c74-129c-4960-ab68-6cab7de088a5", cooldown_minutes: 10 }
				alerts { id: "stockpile-diff-dc-total", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_PROD_STOCKPILE_VLA",
			host_pattern: "vla-\\d{3}",
			dc: "vla",
			mutes {
				project_id: "solomon",
				alerts { id: "solomon-uptime", label_selectors: "service='stockpile', cluster='stockpile_vla'", cooldown_minutes: 10 }
				alerts { project_id: "graphite", id: "ed960c74-129c-4960-ab68-6cab7de088a5", cooldown_minutes: 10 }
			}
		}
	}
	cloud_pre {
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PREPROD_STORAGE",
			host_pattern: "\\d{2}",
			dc: "sas",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "pre-solomon-uptime", label_selectors: "service='stockpile'", cooldown_minutes: 10 }
			}
		}
	}
	cloud_prod {
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_STOCKPILE_SAS",
			host_pattern: "sas-\\d{2}",
			dc: "sas",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "solomon-uptime", label_selectors: "service='stockpile', cluster='stockpile_sas'", cooldown_minutes: 10 }
				alerts { id: "stockpile-diff-dc-total", cooldown_minutes: 10 }
			}
		}
		replicas {
			z2_config_id: "SOLOMON_CLOUD_PROD_STOCKPILE_VLA",
			host_pattern: "vla-\\d{2}",
			dc: "vla",
			mutes {
				project_id: "solomon_cloud",
				alerts { id: "solomon-uptime", label_selectors: "service='stockpile', cluster='stockpile_vla'", cooldown_minutes: 10 }
			}
		}
	}
	cloud_gpn {
		replicas {
			z2_config_id: "SOLOMON_GPN_PROD_STOCKPILE_A",
			host_pattern: "a-\\d{2}",
			dc: "gpn_a",
		}
		replicas {
			z2_config_id: "SOLOMON_GPN_PROD_STOCKPILE_B",
			host_pattern: "b-\\d{2}",
			dc: "gpn_b",
		}
	}
`)

var commonStockpilePackages = apt.NewPackageListMust(
	"yandex-solomon-stockpile")

var internalAndCloudStockpilePackages = apt.NewPackageListMust(
	"yandex-solomon-common-conf",
	"yandex-solomon-stockpile-conf")

var gpnStockpilePackages = apt.NewPackageListMust(
	"gpn-solomon-common-conf",
	"gpn-solomon-stockpile-conf")

var StockpileCmd = &cobra.Command{
	Use:   fmt.Sprintf("stockpile {%s} version [PATTERN]", stockpileClusters.AvailableEnvsStr()),
	Short: "Release Stockpile",
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) < 2 || len(args) > 3 {
			return fmt.Errorf("Usage: " + cmd.Use)
		}
		return nil
	},
	RunE: runStockpileRecipeCmd,
}

const stockpileManagementPort = 4500

func stockpileFreezeShards(ctx context.Context, env hosts.Env, token string, addresses []hostsCommon.Address, freeze bool) error {
	staffonlyClient := staffonly.NewClient(env, token, stockpileManagementPort, true)
	url := fmt.Sprintf("/balancer/allNodeFreeze?flag=%v", freeze)

	for _, addr := range addresses {
		resp, err := staffonlyClient.Get(ctx, addr.Name, url)
		if err != nil {
			log.Printf("cannot get response from %s: %v", addr.Name, err)
		} else {
			// response body is not interesting for this task
			_ = resp.Body.Close()
			if resp.StatusCode == http.StatusOK || resp.StatusCode == http.StatusFound {
				return nil
			}
			log.Printf("got unexpected response status '%s' from '%s'", resp.Status, addr.Name)
		}

		log.Println("will try another address after 1 second")
		utils.Sleep(time.Second)
	}

	if freeze {
		return fmt.Errorf("cannot freeze shards")
	}

	return fmt.Errorf("cannot unfreeze shards")
}

func runStockpileRecipeCmd(cmd *cobra.Command, args []string) error {
	env := hosts.EnvFromStr(args[0])
	if env == hosts.EnvUnknown {
		return fmt.Errorf("unknown environment type: %s", args[0])
	}

	stockpilePackages := commonStockpilePackages

	if env.IsGpn() {
		stockpilePackages = append(stockpilePackages, gpnStockpilePackages...)
	} else {
		stockpilePackages = append(stockpilePackages, internalAndCloudStockpilePackages...)
	}

	version := args[1]
	if err := stockpilePackages.SetVersion(version); err != nil {
		return err
	}

	cluster, err := stockpileClusters.FindCluster(env)
	if err != nil {
		return err
	}

	var pattern string
	if len(args) > 2 {
		pattern = args[2]
	}

	if pattern == "gpn_a" {
		pattern = "SOLOMON_GPN_PROD_STOCKPILE_A"
	} else if pattern == "gpn_b" {
		pattern = "SOLOMON_GPN_PROD_STOCKPILE_B"
	}

	replica, err := cluster.FindReplicaConfig(pattern)
	if err != nil {
		return err
	}

	ctx := context.Background()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return fmt.Errorf("cannot load Z2 API keys: %w", err)
	}

	z2Client := z2.NewClient(apiKeys, env.IsCloud())
	hostnames, err := replica.ResolveHosts(ctx, z2Client, pattern)
	if err != nil {
		return err
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	infraServiceName := "Stockpile"
	if env.IsGpn() {
		infraServiceName += " в реплику " + replica.Dcs[0].String()
	}

	eventMaker := &InfraEventMaker{
		Env:            env,
		ServiceName:    infraServiceName,
		UpdateDuration: time.Minute * 30,
		Dcs:            replica.Dcs,
	}

	event, err := eventMaker.UpdateOrMakeNewOne(ctx, token)
	if err != nil {
		return err
	}

	var muteMaker *MuteMaker = nil
	if replica.Mutes != nil {
		muteMaker = &MuteMaker{
			Env:            eventMaker.Env,
			ServiceName:    eventMaker.ServiceName,
			UpdateDuration: eventMaker.UpdateDuration,
			Dcs:            eventMaker.Dcs,
			Mutes:          replica.Mutes,
		}
	}

	if muteMaker != nil {
		log.Println(color.BoldYellow("[*] SET MUTES"))
		err = muteMaker.UpdateOrMakeNew(ctx, token)
		if err != nil {
			return err
		}
	}

	sshClient := ussh.NewClusterClient(addresses, env.IsCloud(), "logs")
	defer sshClient.Close()

	if !env.IsGpn() {
		log.Println(color.BoldYellow("[*] FREEZE SHARDS"))
		if err = stockpileFreezeShards(ctx, env, token, addresses, true); err != nil {
			return err
		}
	}

	log.Println(color.BoldYellow("[*] DOWNLOAD NEW PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr new && mkdir new && cd new && sudo apt-get update && apt-get download "+stockpilePackages.String())

	log.Println(color.BoldYellow("[*] DOWNLOAD OLD PACKAGES"))
	sshClient.RunParallel(maxParallelism, "rm -fr old "+
		"&& mkdir old && cd old "+
		"&& dpkg-query --showformat='${Package}=${Version}\\n' --show "+stockpilePackages.NamesString()+
		" | grep stable | xargs apt-get download")

	log.Println(color.BoldYellow("[*] INSTALL NEW PACKAGES"))
	result := sshClient.RunParallel(maxParallelism, "/Berkanavt/solomon/stockpile/snapshot-logs && sudo dpkg -i -E new/*.deb && rm -fr new")

	log.Println(color.BoldYellow("[*] SLEEP 3 MINUTES"))
	utils.Sleep(3 * time.Minute)

	if !env.IsGpn() {
		log.Println(color.BoldYellow("[*] UNFREEZE SHARDS"))
		if err = stockpileFreezeShards(ctx, env, token, addresses, false); err != nil {
			return err
		}
	}

	if len(hostnames) > 1 {
		if cli.CanContinueUpdate(result.FailedCount) {
			log.Println(color.BoldYellow("[*] UPDATE Z2 CONFIG"))
			if err = updateZ2Config(ctx, z2Client, replica.Z2ConfigID, stockpilePackages); err != nil {
				return err
			}
		} else {
			log.Println(color.BoldRed("[*] Z2 CONFIG WASN'T UPDATE"))
		}

		log.Println(color.BoldYellow("[*] CLOSE INFRA EVENT"))
		if err = eventMaker.FinishEvent(ctx, token, event); err != nil {
			return err
		}

		if muteMaker != nil {
			log.Println(color.BoldYellow("[*] ADJUST MUTES FINISH TIME"))
			if err = muteMaker.AdjustFinishTime(ctx, token); err != nil {
				return err
			}
		}
	}

	return nil
}
