package cmd

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/conductor"
	hostsCommon "a.yandex-team.ru/solomon/libs/go/hosts"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"context"
	"fmt"
	"github.com/mattn/go-isatty"
	"github.com/spf13/cobra"
	"math/rand"
	"os"
	"strings"
	"time"
)

func init() {
	sshCmd := &cobra.Command{
		Use:   "ssh [flags] {hostname|%conductor-group} command",
		Short: "Execute command via SSH on a single host or on multiple hosts",
		Args: func(cmd *cobra.Command, args []string) error {
			if len(args) != 2 {
				return fmt.Errorf("Usage: " + cmd.Use)
			}
			return nil
		},
		RunE: runSSHCmd,
	}
	rootCmd.AddCommand(sshCmd)
}

type cmdResult struct {
	hostname string
	output   []byte
	err      error
}

func runRemoteCmd(client ussh.Client, address *hostsCommon.Address, cmd string) *cmdResult {
	conn, err := client.Connect(address)
	if err != nil {
		return &cmdResult{hostname: address.Name, err: err}
	}
	defer conn.Close()

	output, err := conn.Run(cmd)
	if err != nil {
		return &cmdResult{hostname: address.Name, err: err}
	}

	return &cmdResult{hostname: address.Name, output: output}
}

func runSSHCmd(cmd *cobra.Command, args []string) error {
	hostname := args[0]
	remoteCmd := args[1]
	sshClient := ussh.NewClient(strings.Contains(hostname, "cloud"))

	if strings.HasPrefix(hostname, "%") {
		group := hostname[1:] // skip '%'
		return multipleHosts(group, sshClient, remoteCmd)
	}

	return singleHost(hostname, sshClient, remoteCmd)
}

func multipleHosts(group string, sshClient ussh.Client, remoteCmd string) error {
	conductorClient := conductor.NewClient()
	hostnames, err := conductorClient.GroupToHosts(context.Background(), group)
	if err != nil {
		return fmt.Errorf("cannot resolve conductor group %s to hosts: %w", group, err)
	}

	addresses, err := hosts.ResolveAddresses(hostnames)
	if err != nil {
		return err
	}

	results := make(chan *cmdResult)
	for _, address := range addresses {
		go func(address *hostsCommon.Address) {
			time.Sleep(time.Duration(rand.Int31n(1000)) * time.Millisecond)
			results <- runRemoteCmd(sshClient, address, remoteCmd)
		}(&address)
	}

	okFormat := "[%s] OK\n"
	errFormat := "[%s] ERROR\n"
	if isatty.IsTerminal(os.Stdout.Fd()) {
		okFormat = color.Green(okFormat)
		errFormat = color.Red(errFormat)
	}

	done := 0
	for result := range results {
		if result.err == nil {
			fmt.Printf(okFormat, result.hostname)
			fmt.Println(string(result.output))
		} else {
			fmt.Printf(errFormat, result.hostname)
			fmt.Println(result.err)
		}

		done++
		if done == len(hostnames) {
			break
		}
	}
	return nil
}

func singleHost(hostname string, sshClient ussh.Client, remoteCmd string) error {
	address, err := hosts.ResolveAddress(hostname)
	if err != nil {
		return err
	}

	result := runRemoteCmd(sshClient, address, remoteCmd)
	if result.err != nil {
		return result.err
	}

	fmt.Println(string(result.output))
	return nil
}
