package cmd

import (
	"bytes"
	"context"
	"fmt"
	"html/template"
	"log"
	"regexp"
	"sort"
	"strings"
	"time"

	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/tracker"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/sandbox"
	"github.com/spf13/cobra"
)

var releaseCheckList = []string{
	"------------------------------------------- DAY 1 -------------------------------------------",
	"Packages (COMMON)",
	"Stockpile (VLA)",
	"Backup (VLA)",
	"NameResolver (VLA)",
	"Coremon (VLA)",
	"Fetcher (VLA)",
	"------------------------------------------- DAY 2 -------------------------------------------",
	"Stockpile (SAS)",
	"Backup (SAS)",
	"NameResolver (SAS)",
	"Coremon (SAS)",
	"Fetcher (SAS)",
	"------------------------------------------- DAY 3 -------------------------------------------",
	"DataProxy",
	"ProjectManager",
	"Alerting",
	"Gateway",
	"Backup (Host Gateway)",
	"------------------------------------------- DAY 4 -------------------------------------------",
	"Packages Cloud (COMMON)",
	"Stockpile Cloud (VLA)",
	"NameResolver Cloud (VLA)",
	"Coremon Cloud (VLA)",
	"Fetcher Cloud (VLA)",
	"Backup Cloud (VLA)",
	"------------------------------------------- DAY 5 -------------------------------------------",
	"Stockpile Cloud (SAS)",
	"NameResolver Cloud (SAS)",
	"Coremon Cloud (SAS)",
	"Fetcher Cloud (SAS)",
	"Backup Cloud (SAS)",
	"------------------------------------------- DAY 6 -------------------------------------------",
	"Alerting Cloud",
	"Gateway Cloud",
	"Backup Cloud (gateway)",
}

var descriptionTemplate = `
Ветка: https://a.yandex-team.ru/arc/history/branches/solomon/{{.Branch}}/arcadia/
Инструкция релиз-инженера: https://wiki.yandex-team.ru/solomon/dev/release/

Пакеты:
#|
{{range .Packages}}
|| {{.Repository}} | {{.CreatedStr}} | {{.PackageStr}} | ((https://sandbox.yandex-team.ru/task/{{.TaskID}}/view {{.TaskID}})) ||
{{end}}
|#

++Чтобы изменения описания не удалялись при автоматических обновлениях тикета, оставьте их ниже этой линии++
----------
{{.Appendix}}`

var reReleaseDay = regexp.MustCompile("^20[0-9]{2}-[0-9]{2}-[0-9]{2}$")
var reTicketKey = regexp.MustCompile("^[A-Z]+-[0-9]+$")
var reTicketSummary = regexp.MustCompile("^.*(20[0-9]{2}-[0-9]{2}-[0-9]{2})$")

const (
	endOfGeneratedDescription = "----------\n"
)

func init() {
	ticketCmd := &cobra.Command{
		Use:   "ticket",
		Short: "Commands to manipulate release ticket",
	}
	createCmd := &cobra.Command{
		Use:   "create [RELEASE_DAY]",
		Short: "Create release ticket in Startrek",
		RunE:  runCreateTicketCmd,
	}
	closeCmd := &cobra.Command{
		Use:   "close [TICKET_KEY]",
		Short: "Close release ticket in Startrek",
		RunE:  runCloseTicketCmd,
	}
	listCmd := &cobra.Command{
		Use:   "list",
		Short: "Find release ticket(s)",
		RunE:  runFindTicketCmd,
	}
	updateCmd := &cobra.Command{
		Use:   "update [TICKET_KEY]",
		Short: "Update release ticket description",
		Args: func(cmd *cobra.Command, args []string) error {
			if len(args) > 1 {
				return fmt.Errorf("Usage: " + cmd.Use)
			}
			return nil
		},
		RunE: runUpdateTicketCmd,
	}
	ticketCmd.AddCommand(createCmd)
	ticketCmd.AddCommand(closeCmd)
	ticketCmd.AddCommand(listCmd)
	ticketCmd.AddCommand(updateCmd)
	rootCmd.AddCommand(ticketCmd)
}

func findActiveIssues(ctx context.Context, trackerClient *tracker.Client) ([]tracker.Issue, error) {
	return trackerClient.FindIssues(ctx, tracker.Filter{
		Queue:      &tracker.SolomonQueue,
		Components: []int{tracker.SolomonReleaseID},
		Statuses:   []string{tracker.StatusOpen, tracker.StatusInProgress},
	})
}

func runFindTicketCmd(*cobra.Command, []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	trackerClient := tracker.NewClient(token)
	issues, err := findActiveIssues(ctx, trackerClient)
	if err != nil {
		return fmt.Errorf("cannot find release ticket: %w", err)
	}

	if len(issues) == 0 {
		log.Println("No open release tickets")
	} else {
		log.Printf("Open release tickets:")
		for i, issue := range issues {
			fmt.Printf("    %d. %s\n", i+1, issue.URL())
		}
	}

	return nil
}

func makeIssueDescription(ctx context.Context, token, branchName, appendix string) (string, error) {
	sandboxClient := sandbox.NewClient(token)
	tasks, err := sandboxClient.ListTasks(ctx, "^SOLOMON.*", branchName)
	if err != nil {
		return "", err
	}

	if len(tasks) == 0 {
		return "", fmt.Errorf("cannot find tasks for packages from new branch in sandbox")
	}

	deployItems, err := newDeployItems(tasks)
	if err != nil {
		return "", err
	}

	deployItems = uniqueDeployItems(deployItems)
	sort.SliceStable(deployItems, func(i, j int) bool {
		return deployItems[i].Package.Name < deployItems[j].Package.Name
	})
	sort.SliceStable(deployItems, func(i, j int) bool {
		return deployItems[i].Created.Before(deployItems[j].Created)
	})
	sort.SliceStable(deployItems, func(i, j int) bool {
		return deployItems[i].Repository < deployItems[j].Repository
	})

	type descriptionInfo struct {
		Branch   string
		Packages []deployItem
		Appendix string
	}

	var buf bytes.Buffer

	tmpl := template.Must(template.New("description").Parse(descriptionTemplate))
	err = tmpl.Execute(&buf, descriptionInfo{
		Branch:   branchName,
		Packages: deployItems,
		Appendix: appendix,
	})

	if err != nil {
		return "", fmt.Errorf("cannot render issue description template: %w", err)
	}
	return buf.String(), nil
}

type deployItem struct {
	Repository string
	Package    apt.Package
	PackageStr string
	Created    time.Time
	CreatedStr string
	TaskID     int64
}

func newDeployItems(tasks []sandbox.Task) ([]deployItem, error) {
	result := make([]deployItem, 0, len(tasks))
	for _, task := range tasks {
		packages, err := apt.NewPackageList(task.Packages...)
		if err != nil {
			return nil, err
		}

		for _, p := range packages {
			item := deployItem{
				Repository: task.Repository,
				Package:    p,
				PackageStr: p.String(),
				Created:    task.Created,
				CreatedStr: task.Created.Format(time.RFC3339),
				TaskID:     task.ID,
			}
			result = append(result, item)
		}
	}
	return result, nil
}

func uniqueDeployItems(items []deployItem) []deployItem {
	type key struct {
		Repository  string
		PackageName string
	}

	itemByKey := make(map[key]deployItem, len(items))
	for _, item := range items {
		k := key{Repository: item.Repository, PackageName: item.Package.Name}
		prev, present := itemByKey[k]
		if !present {
			itemByKey[k] = item
			continue
		}

		compare := item.Package.Version.Compare(prev.Package.Version)
		if compare > 0 {
			itemByKey[k] = item
		}
	}

	result := make([]deployItem, 0, len(itemByKey))
	for _, p := range itemByKey {
		result = append(result, p)
	}

	return result
}

func setCheckList(client *tracker.Client, issueKey string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	checkListItems := make([]tracker.CheckListItem, len(releaseCheckList))
	for i := 0; i < len(releaseCheckList); i++ {
		checkListItems[i] = tracker.CheckListItem{Text: releaseCheckList[i]}
	}

	if err := client.SetCheckListItems(ctx, issueKey, checkListItems); err != nil {
		return fmt.Errorf("cannot set checklist items: %w", err)
	}
	return nil
}

func linkRelatedTickets(client *tracker.Client, issueKey string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 20*time.Second)
	defer cancel()

	issues, err := client.FindIssues(ctx, tracker.Filter{
		Queue:    &tracker.SolomonQueue,
		Statuses: []string{tracker.StatusCommited, tracker.StatusReadyForRelease},
	})
	if err != nil {
		return fmt.Errorf("cannot find tickets in to link with release ticket: %w", err)
	}

	if len(issues) == 0 {
		log.Println("No ready for release or commited tickets found")
		return nil
	}

	sort.Slice(issues, func(i, j int) bool {
		return *issues[i].Key < *issues[j].Key
	})

	log.Println("Link next tickets with release ticket:")
	for i, issue := range issues {
		fmt.Printf("    %d. %s %s\n", i+1, color.Yellow(*issue.Key), issue.Summary)
		if err = client.CreateLink(ctx, issueKey, *issue.Key); err != nil {
			return fmt.Errorf("cannot create link between %s and %s: %w", issueKey, *issue.Key, err)
		}
	}

	return nil
}

func runCreateTicketCmd(_ *cobra.Command, args []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	trackerClient := tracker.NewClient(token)
	myself, err := trackerClient.Myself(ctx)
	if err != nil {
		return err
	}

	var releaseDay string
	if len(args) == 0 {
		releaseDay = time.Now().Format("2006-01-02")
	} else if reReleaseDay.MatchString(args[0]) {
		releaseDay = args[0]
	} else {
		return fmt.Errorf("use 2006-01-02 format for release date")
	}
	branchName := "stable-" + releaseDay
	description, err := makeIssueDescription(ctx, token, branchName, "")
	if err != nil {
		return err
	}

	issue := &tracker.Issue{
		Queue:       tracker.SolomonQueue,
		Summary:     "Релиз Solomon " + releaseDay,
		Assignee:    tracker.Assignee{ID: myself.Login},
		Components:  []tracker.Component{tracker.SolomonRelease},
		Description: description,
	}

	if err = trackerClient.CreateIssue(ctx, issue); err != nil {
		return fmt.Errorf("cannot create ticket: %w", err)
	}
	log.Printf("Created ticket: %s", color.Green(issue.URL()))
	if err = trackerClient.OpenIssue(ctx, *issue.Key); err != nil {
		return fmt.Errorf("cannot open ticket: %w", err)
	}
	if err = setCheckList(trackerClient, *issue.Key); err != nil {
		return err
	}
	return linkRelatedTickets(trackerClient, *issue.Key)
}

func runCloseTicketCmd(_ *cobra.Command, args []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 2*time.Minute)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	var issue *tracker.Issue

	trackerClient := tracker.NewClient(token)
	if len(args) == 0 {
		issues, err := findActiveIssues(ctx, trackerClient)
		if err != nil {
			return fmt.Errorf("listing issues failed: %w", err)
		}
		if len(issues) == 0 {
			fmt.Println("No active release tickets found, try specifying ticket key")
			return nil
		}
		if len(issues) > 1 {
			fmt.Println("There are more than one active release tickets:")
			for i, issue := range issues {
				fmt.Printf("  %d. %s\n", i+1, issue.URL())
			}
			fmt.Println("Please add an argument to specify which ticket you want to close.")
			return nil
		}
		issue = &issues[0]
	} else {
		issue, err = trackerClient.GetIssue(ctx, args[0])
		if err != nil {
			return fmt.Errorf("failed to get ticket by key: %w", err)
		}
	}

	if !tracker.IsSolomonReleaseIssue(issue) {
		return fmt.Errorf("ticket %s has no 'Release' component", *issue.Key)
	}

	log.Println("Closing release ticket", issue.URL())
	{
		log.Println("  * ensure that all checks are completed")
		checkList, err := trackerClient.GetCheckListItems(ctx, *issue.Key)
		if err != nil {
			return fmt.Errorf("cannot get list of checks: %w", err)
		}

		if len(checkList) == 0 {
			return fmt.Errorf("release ticket without check list")
		}

		for _, item := range checkList {
			if !item.Checked {
				return fmt.Errorf(
					"%s is not checked yet. Make sure that all checks in ticket %s are checked",
					item.Text, *issue.Key)
			}
		}
	}

	{
		log.Println("  * closing all linked tickets")
		links, err := trackerClient.GetLinks(ctx, *issue.Key)
		if err != nil {
			return fmt.Errorf("cannot get list of linked tickets: %w", err)
		}

		for _, link := range links {
			status := link.Status.Key
			key := link.Object.Key

			switch status {
			case tracker.StatusClosed:
				fmt.Printf("    - skip  [%s] %s (%s)\n", key, link.Object.Display, link.Assignee.ID)
			case tracker.StatusReadyForRelease:
				fmt.Printf("    - close [%s] %s (%s)\n", key, link.Object.Display, link.Assignee.ID)
				if err = trackerClient.ResolveIssue(ctx, key); err != nil {
					return fmt.Errorf("cannot close ticket %s: %w", key, err)
				}
			default:
				return fmt.Errorf("cannot close ticket %s (%s), status is not ReadFroRelease", key, link.Status.Key)
			}
		}
	}

	{
		log.Println("  * closing release ticket")
		if err = trackerClient.ResolveIssue(ctx, *issue.Key); err != nil {
			return fmt.Errorf("cannot close release ticket %s: %w", *issue.Key, err)
		}
	}

	log.Println("Done")
	return nil
}

func runUpdateTicketCmd(_ *cobra.Command, args []string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return err
	}

	trackerClient := tracker.NewClient(token)

	var issue *tracker.Issue
	var releaseDay, argIssue string

	if len(args) == 1 {
		if !reTicketKey.MatchString(args[0]) {
			return fmt.Errorf("use XXXXX-1234 format for ticket")
		}
		argIssue = args[0]
	}
	if argIssue == "" {
		issues, err := findActiveIssues(ctx, trackerClient)
		if err != nil {
			return fmt.Errorf("listing issues failed: %w", err)
		}
		if len(issues) == 0 {
			return fmt.Errorf("no active release tickets found, try specifying ticket key")
		}
		if len(issues) > 1 {
			return fmt.Errorf("too many active release tickets found, try specifying ticket key")
		}
		issue = &issues[0]
	} else {
		issue, err = trackerClient.GetIssue(ctx, argIssue)
		if err != nil {
			return fmt.Errorf("failed to get ticket by key: %w", err)
		}
	}

	match := reTicketSummary.FindStringSubmatch(issue.Summary)
	releaseDay = match[1]

	endOfGenIdx := strings.Index(issue.Description, endOfGeneratedDescription)
	var appendix string
	if endOfGenIdx == -1 {
		fmt.Println(color.Yellow("Magic comment was missing in the ticket description, assuming nothing to preserve"))
		appendix = ""
	} else {
		appendix = issue.Description[endOfGenIdx+len(endOfGeneratedDescription):]
	}

	branchName := "stable-" + releaseDay
	description, err := makeIssueDescription(ctx, token, branchName, appendix)
	if err != nil {
		return err
	}

	newIssue := &tracker.Issue{
		Key:         issue.Key,
		Version:     issue.Version,
		Queue:       issue.Queue,
		Summary:     issue.Summary,
		Assignee:    issue.Assignee,
		Components:  issue.Components,
		Description: description,
	}

	if err = trackerClient.UpdateIssue(ctx, newIssue); err != nil {
		return fmt.Errorf("cannot update ticket: %w", err)
	}
	log.Printf("Updated ticket: %s", color.Green(issue.URL()))
	return nil
}
