package cmd

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"a.yandex-team.ru/solomon/tools/release/internal/oauth"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"strings"
	"time"
)

func init() {
	z2Cmd := &cobra.Command{
		Use:   "z2 configId package1=version1 [package2=version2, ...]",
		Short: "Update packages in specified Z2 config",
		Args: func(cmd *cobra.Command, args []string) error {
			if len(args) < 2 {
				return fmt.Errorf("Usage: " + cmd.Use)
			}
			return nil
		},
		RunE: runZ2Cmd,
	}
	rootCmd.AddCommand(z2Cmd)
}

func runZ2Cmd(cmd *cobra.Command, args []string) error {
	configID := args[0]
	pkgs := apt.PackageList{}

	for i := 1; i < len(args); i++ {
		pkg, err := apt.ParsePackage(args[i])
		if err != nil {
			return err
		}
		if !pkg.HasVersion() {
			return fmt.Errorf("package %s has no version", pkg.Name)
		}
		pkgs = append(pkgs, *pkg)
	}

	apiKeys, err := loadAPIKeys()
	if err != nil {
		return err
	}

	z2Client := z2.NewClient(apiKeys, strings.Contains(configID, "_CLOUD_"))

	{
		ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
		defer cancel()

		fmt.Printf("Edit Z2 config %s", configID)
		if err = z2Client.EditItems(ctx, configID, pkgs); err != nil {
			fmt.Println(color.Red("\t[FAILED]"))
			return fmt.Errorf("cannot edit items in %s: %w", configID, err)
		}
		fmt.Println(color.Green("\t[OK]"))

		fmt.Printf("Update Z2 config %s", configID)
		if err = z2Client.Update(ctx, configID); err != nil {
			fmt.Println(color.Red("\t[FAILED]"))
			return fmt.Errorf("cannot update config %s: %w", configID, err)
		}
		fmt.Println(color.Green("\t[OK]"))
	}

	fmt.Printf("Go to %s to see update status\n", z2Client.ControlPanelURL(configID))
	for i := 0; ; i++ {
		status, err := z2Client.UpdateStatus(context.Background(), configID)
		if err != nil {
			return fmt.Errorf("cannot get update status in %s: %w", configID, err)
		}

		fmt.Printf("\033[100DUpdate status %s (%d)", status.UpdateStatus, i)
		if status.IsFinished() {
			fmt.Print("\033[100D")
			status.Print(configID)
			break
		}

		time.Sleep(2 * time.Second)
	}

	return nil
}

func loadAPIKeys() (z2.APIKeys, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	token, err := oauth.GetMyToken(ctx)
	if err != nil {
		return nil, fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}

	apiKeys, err := z2.LoadAPIKeys(ctx, token)
	if err != nil {
		return nil, fmt.Errorf("cannot load Z2 API keys: %w", err)
	}
	return apiKeys, nil
}
