package apt

import (
	"fmt"
	"sort"
	"strconv"
	"strings"
)

type Package struct {
	Name    string   `json:"name"`
	Version *Version `json:"version"`
}

func (p *Package) String() string {
	if p.Version == nil {
		return p.Name
	}
	return p.Name + "=" + p.Version.String()
}

func (p *Package) HasVersion() bool {
	return p.Version != nil
}

func (p *Package) Equal(q *Package) bool {
	if p.Name != q.Name {
		return false
	}
	if (p.Version == nil && q.Version != nil) || (p.Version != nil && q.Version == nil) {
		return false
	}
	return p.Version == nil || q.Version.Compare(q.Version) == 0
}

func ParsePackage(str string) (*Package, error) {
	if idx := strings.IndexByte(str, '='); idx != -1 {
		name := str[0:idx]
		version, err := MakeVersion(str[idx+1:])
		if err != nil {
			return nil, err
		}
		return &Package{Name: name, Version: version}, nil
	}
	return &Package{Name: str}, nil
}

// ====================================================================================

type PackageList []Package

func NewPackageList(names ...string) (PackageList, error) {
	pl := make(PackageList, 0, len(names))
	for _, name := range names {
		pkg, err := ParsePackage(name)
		if err != nil {
			return nil, err
		}
		pl = append(pl, *pkg)
	}
	return pl, nil
}

func NewPackageListMust(names ...string) PackageList {
	pl, err := NewPackageList(names...)
	if err != nil {
		panic(err)
	}
	return pl
}

func (pl PackageList) SetVersion(versionString string) error {
	for i := 0; i < len(pl); i++ {
		version, err := MakeVersion(versionString)
		if err != nil {
			return err
		}
		pl[i].Version = version
	}
	return nil
}

func (pl PackageList) SetVersionMust(versionString string) {
	err := pl.SetVersion(versionString)
	if err != nil {
		panic(err)
	}
}

func (pl PackageList) String() string {
	var sb strings.Builder
	for i, p := range pl {
		if i > 0 {
			sb.WriteByte(' ')
		}
		sb.WriteString(p.String())
	}
	return sb.String()
}

func (pl PackageList) NamesString() string {
	var sb strings.Builder
	for i, p := range pl {
		if i > 0 {
			sb.WriteByte(' ')
		}
		sb.WriteString(p.Name)
	}
	return sb.String()
}

func (pl PackageList) UniqLatestVersions() PackageList {
	newPkgs := make(map[string]*Package, len(pl))
	for i := 0; i < len(pl); i++ {
		if p, ok := newPkgs[pl[i].Name]; ok {
			if pl[i].Version.Compare(p.Version) > 0 {
				newPkgs[pl[i].Name] = &pl[i]
			}
		} else {
			newPkgs[pl[i].Name] = &pl[i]
		}
	}
	newList := make(PackageList, 0, len(newPkgs))
	for _, p := range newPkgs {
		newList = append(newList, *p)
	}
	sort.Slice(newList, func(i, j int) bool {
		return strings.Compare(newList[i].Name, newList[j].Name) < 0
	})
	return newList
}

// ====================================================================================
//
// As per
// https://www.debian.org/doc/debian-policy/ch-controlfields.html#s-f-version
// https://www.debian.org/doc/manuals/debmake-doc/ch05.en.html#name-version
//
// v.Compare(u) >  0 ==> v >  u
// v.Compare(u) == 0 ==> v == u
// v.Compare(u) <  0 ==> v <  u

type Version struct {
	Epoch           string
	UpstreamVersion []string
	DebianRevision  []string
	// ints
	epochInt     int
	versionInts  []int
	revisionInts []int
}

func (v Version) MarshalJSON() ([]byte, error) {
	return []byte(`"` + v.String() + `"`), nil
}

func (v *Version) UnmarshalJSON(b []byte) error {
	if len(b) < 2 || b[0] != '"' || b[len(b)-1] != '"' {
		return fmt.Errorf("bad data to unmarshal package version, %s", string(b))
	}
	nv, err := MakeVersion(string(b[1 : len(b)-1]))
	if err != nil {
		return err
	}
	v.Epoch = nv.Epoch
	v.UpstreamVersion = nv.UpstreamVersion
	v.DebianRevision = nv.DebianRevision
	v.epochInt = nv.epochInt
	v.versionInts = nv.versionInts
	v.revisionInts = nv.revisionInts
	return nil
}

func MakeVersion(s string) (*Version, error) {
	var err error
	v := &Version{}
	idxMin, idxMaxVer, idxMinRev := 0, len(s), len(s)

	if idx := strings.IndexByte(s, ':'); idx >= 0 {
		v.Epoch = s[:idx]
		if v.epochInt, err = strconv.Atoi(v.Epoch); err != nil {
			return nil, err
		}
		idxMin = idx + 1
	}
	if idx := strings.LastIndexByte(s, '-'); idx >= 0 {
		idxMaxVer, idxMinRev = idx, idx+1
	}

	splitString := func(idxPart, idxMax int, hyphenOk bool) ([]string, []int) {
		r := []string{}
		rInts := []int{}
		isNum := false
		for idx := idxPart; idx < idxMax; idx++ {
			c := s[idx]
			if c >= '0' && c <= '9' {
				if !isNum {
					isNum = true
					r = append(r, s[idxPart:idx])
					idxPart = idx
				}
			} else if (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || strings.IndexByte(".+~", c) >= 0 || (c == '-' && hyphenOk) {
				if isNum {
					isNum = false
					r = append(r, s[idxPart:idx])
					i, _ := strconv.Atoi(s[idxPart:idx])
					rInts = append(rInts, i)
					idxPart = idx
				}
			} else {
				return nil, nil
			}
		}
		r = append(r, s[idxPart:idxMax])
		if isNum && idxPart < idxMax {
			i, _ := strconv.Atoi(s[idxPart:idxMax])
			rInts = append(rInts, i)
		}
		return r, rInts
	}

	v.UpstreamVersion, v.versionInts = splitString(idxMin, idxMaxVer, idxMaxVer != idxMinRev)
	v.DebianRevision, v.revisionInts = splitString(idxMinRev, len(s), false)

	if v.UpstreamVersion == nil || v.DebianRevision == nil {
		return nil, fmt.Errorf("bad version %s", s)
	}
	return v, nil
}

func (v *Version) String() string {
	s := ""
	if v.Epoch != "" {
		s += v.Epoch + ":"
	}
	s += strings.Join(v.UpstreamVersion, "")
	if len(v.DebianRevision) > 1 || v.DebianRevision[0] != "" {
		s += "-" + strings.Join(v.DebianRevision, "")
	}
	return s
}

func versionListsCompare(a, b []string, x, y []int) int {
	min := func(a, b int) int {
		if a < b {
			return a
		}
		return b
	}
	strComp := func(a, b string) int {
		// The lexical comparison is a comparison of ASCII values modified so that
		// all the letters sort earlier than all the non-letters and so that
		// a tilde sorts before anything, even the end of a part.
		minLen := min(len(a), len(b))
		for idx := 0; idx < minLen; idx++ {
			ca, cb := a[idx], b[idx]
			if ca == cb {
				continue
			} else if ca == '~' {
				return -1
			} else if cb == '~' {
				return 1
			}
			aIsLetter := (ca >= 'a' && ca <= 'z') || (cb >= 'A' && cb <= 'Z')
			bIsLetter := (cb >= 'a' && cb <= 'z') || (cb >= 'A' && cb <= 'Z')
			if aIsLetter && !bIsLetter {
				return -1
			} else if !aIsLetter && bIsLetter {
				return 1
			}
			return int(ca) - int(cb)
		}
		if len(a) != len(b) {
			if minLen == len(a) && b[minLen] == '~' {
				return 1
			}
			if minLen == len(b) && a[minLen] == '~' {
				return -1
			}
		}
		return len(a) - len(b)
	}

	minLen := min(len(a), len(b))
	isDigit := false
	for idx := 0; idx < minLen; idx++ {
		if isDigit {
			if d := x[idx/2] - y[idx/2]; d != 0 {
				return d
			}
		} else {
			if d := strComp(a[idx], b[idx]); d != 0 {
				return d
			}
		}
		isDigit = !isDigit
	}
	// Case 1234~
	if !isDigit && len(a) != len(b) {
		if minLen == len(a) && b[minLen][0] == '~' {
			return 1
		}
		if minLen == len(b) && a[minLen][0] == '~' {
			return -1
		}
	}
	return len(a) - len(b)
}

func (v *Version) Compare(u *Version) int {
	if d := v.epochInt - u.epochInt; d != 0 {
		return d
	}
	if c := versionListsCompare(v.UpstreamVersion, u.UpstreamVersion, v.versionInts, u.versionInts); c != 0 {
		return c
	}
	return versionListsCompare(v.DebianRevision, u.DebianRevision, v.revisionInts, u.revisionInts)
}
