package hosts

import (
	hostsPB "a.yandex-team.ru/solomon/tools/release/internal/hosts/proto"
	"a.yandex-team.ru/solomon/tools/release/internal/z2"
	"context"
	"fmt"
	"github.com/golang/protobuf/proto"
	"log"
	"regexp"
	"strings"
	"time"
)

type Alert struct {
	ID             string
	LabelSelectors string
	CoolDown       time.Duration
	ProjectID      string
}

type MuteList struct {
	ProjectID string
	Alerts    []Alert
}

type Replica struct {
	Z2ConfigID  string
	HostPattern *regexp.Regexp
	Dcs         []Dc
	Mutes       *MuteList
}

type Cluster struct {
	Env      Env
	Replicas []Replica
}

type ClusterList []Cluster

func (c *Cluster) FindReplicaConfig(pattern string) (*Replica, error) {
	if pattern == "" {
		// (1) empty pattern can be valid only if there is only one Z2 config
		if len(c.Replicas) == 1 {
			return &c.Replicas[0], nil
		}

		ids := make([]string, 0, 2)
		for _, cfg := range c.Replicas {
			ids = append(ids, cfg.Z2ConfigID)
		}
		return nil, fmt.Errorf("you must provide pattern to choose from Z2 configs: %s", strings.Join(ids, ", "))
	}

	// (2) try to use pattern as host pattern and find concrete host
	for _, cfg := range c.Replicas {
		if cfg.HostPattern.MatchString(pattern) {
			return &cfg, nil
		}
	}

	// (3) try to use pattern as Z2 config ID and find concrete one
	var foundCfg *Replica
	patternUpper := strings.ToUpper(pattern)
	for _, cfg := range c.Replicas {
		if strings.Contains(cfg.Z2ConfigID, patternUpper) {
			if foundCfg != nil {
				return nil, fmt.Errorf(
					"pattern %s matches more than one Z2 config: %s and %s",
					pattern, foundCfg.Z2ConfigID, cfg.Z2ConfigID)
			}
			tmp := cfg
			foundCfg = &tmp
		}
	}

	if foundCfg == nil {
		return nil, fmt.Errorf("cannot find hosts in %s by pattern %s", c.Env, pattern)
	}
	return foundCfg, nil
}

func (c *Replica) ResolveHosts(ctx context.Context, client z2.Client, pattern string) ([]string, error) {
	workers, err := client.Workers(ctx, c.Z2ConfigID)
	if err != nil {
		return nil, fmt.Errorf("cannot load workers list from Z2 config %s: %w", c.Z2ConfigID, err)
	}

	if pattern != "" && c.HostPattern.MatchString(pattern) {
		for _, worker := range workers.Workers {
			if strings.Contains(worker, pattern) {
				return []string{worker}, nil
			}
		}
	}

	return workers.Workers, nil
}

func ParseClusterListProtoText(text string) ClusterList {
	pbClusterList := &hostsPB.ClusterList{}
	err := proto.UnmarshalText(text, pbClusterList)
	if err != nil {
		log.Fatalf("cannot parse proto text '%s' to cluster list: %v", text, err)
	}

	result := make(ClusterList, 0, 5)
	if pbClusterList.Test != nil {
		result = append(result, makeCluster(EnvTest, pbClusterList.Test))
	}
	if pbClusterList.Pre != nil {
		result = append(result, makeCluster(EnvPre, pbClusterList.Pre))
	}
	if pbClusterList.Prod != nil {
		result = append(result, makeCluster(EnvProd, pbClusterList.Prod))
	}
	if pbClusterList.CloudPre != nil {
		result = append(result, makeCluster(EnvCloudPre, pbClusterList.CloudPre))
	}
	if pbClusterList.CloudProd != nil {
		result = append(result, makeCluster(EnvCloudProd, pbClusterList.CloudProd))
	}
	if pbClusterList.CloudGpn != nil {
		result = append(result, makeCluster(EnvCloudGpn, pbClusterList.CloudGpn))
	}
	return result
}

func makeCluster(env Env, pbCluster *hostsPB.Cluster) Cluster {
	replicas := make([]Replica, len(pbCluster.Replicas))
	for i, pbReplica := range pbCluster.Replicas {
		replicas[i] = makeReplica(pbReplica)
	}
	return Cluster{Env: env, Replicas: replicas}
}

func makeReplica(pbReplica *hostsPB.Replica) Replica {
	hostPattern, err := regexp.Compile(pbReplica.HostPattern)
	if err != nil {
		log.Fatalf("cannot compile host patter '%s' in Z2 config %s: %v", pbReplica.HostPattern, pbReplica.Z2ConfigId, err)
	}
	dcs, err := ParseDcs(pbReplica.Dc)
	if err != nil {
		log.Fatalf("in Z2 config '%s': %v", pbReplica.Z2ConfigId, err)
	}
	mutes, err := ParseMutes(pbReplica.Mutes)
	if err != nil {
		log.Fatalf("in Z2 config '%s': %v", pbReplica.Z2ConfigId, err)
	}
	return Replica{Z2ConfigID: pbReplica.Z2ConfigId, HostPattern: hostPattern, Dcs: dcs, Mutes: mutes}
}

func (cl ClusterList) AvailableEnvsStr() string {
	names := make([]string, len(cl))
	for i, c := range cl {
		names[i] = c.Env.String()
	}
	return strings.Join(names, "|")
}

func (cl ClusterList) FindCluster(env Env) (*Cluster, error) {
	for _, cluster := range cl {
		if cluster.Env == env {
			return &cluster, nil
		}
	}
	return nil, fmt.Errorf("unknown cluster, available are: %s", cl.AvailableEnvsStr())
}
