package hosts

import (
	"fmt"
	"strings"
)

type Dc int

const (
	DcUnknown Dc = iota
	DcSas
	DcVla
	DcMan
	DcMyt
	DcIva
	DcGpnA
	DcGpnB
)

var DcNames = []string{
	"unknown",
	"sas",
	"vla",
	"man",
	"myt",
	"iva",
	"gpn_a",
	"gpn_b",
}

func (dc Dc) String() string {
	return DcNames[dc]
}

func DcFromStr(str string) Dc {
	switch strings.ToLower(str) {
	case "sas":
		return DcSas
	case "vla":
		return DcVla
	case "man":
		return DcMan
	case "myt":
		return DcMyt
	case "iva":
		return DcIva
	case "gpn_a":
		return DcGpnA
	case "gpn_b":
		return DcGpnB
	default:
		return DcUnknown
	}
}

func ParseDcs(dcsStr string) ([]Dc, error) {
	dcsStr = strings.TrimSpace(dcsStr)
	if len(dcsStr) == 0 {
		return []Dc{}, nil
	}

	dcs := make([]Dc, 0, 5)
	for _, dcStr := range strings.Split(dcsStr, ",") {
		dcStr = strings.TrimSpace(dcStr)
		if dc := DcFromStr(dcStr); dc != DcUnknown {
			dcs = append(dcs, dc)
		} else {
			return nil, fmt.Errorf("unknown DC: %s", dcStr)
		}
	}
	return dcs, nil
}
