package infra

import (
	"a.yandex-team.ru/solomon/libs/go/uhttp"
	"context"
	"fmt"
	"strings"
	"time"
)

const (
	baseURL     = "https://infra-api.yandex-team.ru/v1"
	httpTimeout = 5 * time.Second
)

type Client struct {
	httpClient *uhttp.Client
}

func NewClient(token string) *Client {
	return &Client{httpClient: uhttp.NewClient(baseURL, "solomon-release-tool", uhttp.OAuthToken(token), httpTimeout, true)}
}

func (c *Client) CreateEvent(ctx context.Context, event *Event) error {
	req, err := c.httpClient.NewPostJSONRequest(ctx, "/events", event)
	if err != nil {
		return err
	}

	var resp struct {
		ID int `json:"id"`
	}

	err = c.httpClient.SendJSONRequest(req, &resp)
	if err != nil {
		return err
	}

	event.ID = resp.ID
	return nil
}

func (c *Client) FindCurrentEvents(ctx context.Context, srvEnv ServiceEnv, title string) (*Event, error) {
	nowUnix := time.Now().Unix()
	url := fmt.Sprintf("/events?serviceId=%d&environmentId=%d&from=%d&to=%d",
		srvEnv.ServiceID,
		srvEnv.EnvID,
		nowUnix,
		nowUnix)

	req, err := c.httpClient.NewGetRequest(ctx, url)
	if err != nil {
		return nil, err
	}

	// Infra accepts json with camelCase but responds with snake_case
	var resp []struct {
		ID            int       `json:"id"`
		ServiceID     ServiceID `json:"service_id"`
		EnvironmentID EnvID     `json:"environment_id"`
		Title         string    `json:"title"`
		Description   string    `json:"description"`
		Type          string    `json:"type"`
		Severity      string    `json:"severity"`
		StartTime     int64     `json:"start_time"`
		FinishTime    int64     `json:"finish_time"`
		Tickets       *string   `json:"tickets,omitempty"`
		Sas           *bool     `json:"sas,omitempty"`
		Vla           *bool     `json:"vla,omitempty"`
		Man           *bool     `json:"man,omitempty"`
		Myt           *bool     `json:"myt,omitempty"`
		Iva           *bool     `json:"iva,omitempty"`
	}

	err = c.httpClient.SendJSONRequest(req, &resp)
	if err != nil {
		return nil, err
	}

	for _, e := range resp {
		if strings.Contains(e.Title, title) {
			event := &Event{
				ID:                     e.ID,
				ServiceID:              e.ServiceID,
				EnvironmentID:          e.EnvironmentID,
				Title:                  e.Title,
				Description:            e.Description,
				Type:                   e.Type,
				Severity:               e.Severity,
				StartTime:              e.StartTime,
				FinishTime:             e.FinishTime,
				SendEmailNotifications: false, // intentionally set to false
				Tickets:                e.Tickets,
				Sas:                    e.Sas,
				Vla:                    e.Vla,
				Man:                    e.Man,
				Myt:                    e.Myt,
				Iva:                    e.Iva,
			}
			return event, nil
		}
	}

	return nil, nil
}

func (c *Client) UpdateEvent(ctx context.Context, event *Event) error {
	if event.ID == 0 {
		return fmt.Errorf("cannot update event with unknown id")
	}

	url := fmt.Sprintf("/events/%d", event.ID)
	req, err := c.httpClient.NewPutJSONRequest(ctx, url, event)
	if err != nil {
		return err
	}

	return c.httpClient.SendJSONRequest(req, nil)
}
