package infra

import (
	"a.yandex-team.ru/solomon/tools/release/internal/hosts"
	"fmt"
	"strings"
	"time"
)

type ServiceID int
type EnvID int

type ServiceEnv struct {
	ServiceID ServiceID
	EnvID     EnvID
}

const (
	serviceSolomon      ServiceID = 4
	serviceSolomonCloud ServiceID = 309
	serviceGrafana      ServiceID = 342
	serviceGraphite     ServiceID = 343
)

var services = map[string]map[hosts.Env]*ServiceEnv{
	"solomon": {
		hosts.EnvTest:      {ServiceID: serviceSolomon, EnvID: 11},
		hosts.EnvPre:       {ServiceID: serviceSolomon, EnvID: 11},
		hosts.EnvProd:      {ServiceID: serviceSolomon, EnvID: 10},
		hosts.EnvCloudPre:  {ServiceID: serviceSolomonCloud, EnvID: 429},
		hosts.EnvCloudProd: {ServiceID: serviceSolomonCloud, EnvID: 428},
		hosts.EnvCloudGpn:  {ServiceID: serviceSolomonCloud, EnvID: 2936},
	},
	"grafana": {
		hosts.EnvProd: {ServiceID: serviceGrafana, EnvID: 542},
	},
	"graphite": {
		// TODO: add graphite environments
		// "bs": 543,
		// "gr": 544,
		// "ps": 545,
	},
}

func ListServices() []string {
	res := make([]string, len(services))
	i := 0
	for service := range services {
		res[i] = service
		i++
	}
	return res
}

func FindServiceEnv(service string, env hosts.Env) (*ServiceEnv, error) {
	envMap, ok := services[service]
	if !ok {
		return nil, fmt.Errorf("unknown service, valid values are: %v", strings.Join(ListServices(), ", "))
	}
	serviceEnv, ok := envMap[env]
	if !ok {
		envs := make([]string, len(envMap))
		i := 0
		for envHosts := range envMap {
			envs[i] = envHosts.String()
			i++
		}
		return nil, fmt.Errorf("env '%s' is not configured for service '%s', valid envs are: %v", service,
			env.String(), strings.Join(envs, ", "))
	}
	return serviceEnv, nil
}

func (s *ServiceEnv) NewEvent(title, description string, duration time.Duration, ticket string, dcs []hosts.Dc) *Event {
	now := time.Now()
	event := &Event{
		ServiceID:              s.ServiceID,
		EnvironmentID:          s.EnvID,
		Title:                  title,
		Description:            description,
		Type:                   "maintenance",
		Severity:               "minor",
		StartTime:              now.Unix(),
		FinishTime:             now.Add(duration).Unix(),
		SendEmailNotifications: true,
	}

	if len(ticket) != 0 {
		event.Tickets = &ticket
	}

	flag := true
	for _, dc := range dcs {
		switch dc {
		case hosts.DcSas:
			event.Sas = &flag
		case hosts.DcVla:
			event.Vla = &flag
		case hosts.DcMan:
			event.Man = &flag
		case hosts.DcMyt:
			event.Myt = &flag
		case hosts.DcIva:
			event.Iva = &flag
		case hosts.DcGpnA:
			event.GpnA = &flag
		case hosts.DcGpnB:
			event.GpnB = &flag
		default:
			panic("used unknown DC")
		}
	}

	return event
}
