package mutes

import (
	"a.yandex-team.ru/solomon/libs/go/uhttp"
	"context"
	"fmt"
	"time"
)

const (
	baseURL     = "https://solomon.yandex-team.ru/api/v2"
	httpTimeout = 5 * time.Second
)

type Client struct {
	httpClient *uhttp.Client
}

func NewClient(token string) *Client {
	return &Client{httpClient: uhttp.NewClient(baseURL, "solomon-release-tool", uhttp.OAuthToken(token), httpTimeout, true)}
}

func (c *Client) CreateMute(ctx context.Context, mute *Mute) error {
	url := fmt.Sprintf("/projects/%s/mutes", mute.ProjectID)
	req, err := c.httpClient.NewPostJSONRequest(ctx, url, mute)
	if err != nil {
		return err
	}

	var resp Mute

	err = c.httpClient.SendJSONRequest(req, &resp)
	if err != nil {
		return err
	}

	*mute = resp
	return nil
}

func (c *Client) UpdateMute(ctx context.Context, mute *Mute) error {
	if mute.ID == nil {
		return fmt.Errorf("cannot update mute with unknown id")
	}

	url := fmt.Sprintf("/projects/%s/mutes/%s", mute.ProjectID, *mute.ID)
	req, err := c.httpClient.NewPutJSONRequest(ctx, url, mute)
	if err != nil {
		return err
	}

	return c.httpClient.SendJSONRequest(req, nil)
}

func (c *Client) FillExistingMutesID(ctx context.Context, mutes []Mute) error {
	if len(mutes) == 0 {
		return nil
	}

	projects := make(map[string]string)

	for _, mute := range mutes {
		projects[mute.ProjectID] = mute.ProjectID
	}

	getKey := func(d Mute) string {
		return d.ProjectID + "\t" + d.Type.Selectors.AlertSelector + "\t" + d.Type.Selectors.LabelSelectors
	}

	selectorsToIndex := make(map[string]int, len(mutes))
	for i, mute := range mutes {
		selectorsToIndex[getKey(mute)] = i
	}

	description := mutes[0].Description
	ticketID := mutes[0].TicketID
	for _, mute := range mutes {
		if mute.TicketID != ticketID || mute.Description != description {
			return fmt.Errorf("expected single ticketID and description for all mutes")
		}
	}

	for k := range projects {
		projectID := k
		pageToken := ""
		for {
			// TODO: add filter by ticket and description, for now just full scan
			url := fmt.Sprintf("/projects/%s/mutes?pageSize=100&pageToken=%s", projectID, pageToken)

			req, err := c.httpClient.NewGetRequest(ctx, url)
			if err != nil {
				return err
			}

			var resp struct {
				Items         []Mute  `json:"items"`
				NextPageToken *string `json:"nextPageToken"`
			}

			err = c.httpClient.SendJSONRequest(req, &resp)
			if err != nil {
				return err
			}
			for _, item := range resp.Items {
				if item.TicketID == ticketID && item.Description == description {
					key := getKey(item)
					if index, ok := selectorsToIndex[key]; ok {
						if mutes[index].ID == nil {
							mutes[index].ID = item.ID
						}
						if mutes[index].Version == nil {
							mutes[index].Version = item.Version
						}
						if mutes[index].From == nil {
							mutes[index].From = item.From
						}
					}
				}
			}

			if resp.NextPageToken == nil || *resp.NextPageToken == "" {
				break
			}
			pageToken = *resp.NextPageToken
			resp.NextPageToken = nil
		}
	}

	return nil
}
