package sandbox

import (
	"context"
	"fmt"
	"log"
	"regexp"
	"sort"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/solomon/libs/go/uhttp"
)

const (
	baseURL        = "https://sandbox.yandex-team.ru/api/v1.0"
	httpTimeout    = 5 * time.Second
	retryDelay     = 2 * time.Second
	taskStartTries = 10
)

type Client struct {
	httpClient *uhttp.Client
}

func NewClient(token string) *Client {
	return &Client{httpClient: uhttp.NewClient(baseURL, "solomon-release-tool", uhttp.OAuthToken(token), httpTimeout, true)}
}

func (c *Client) ListTasks(ctx context.Context, descRe, branchName string) ([]Task, error) {
	now := time.Now()
	to := now.Format("2006-01-02T15:04:05Z")
	from := now.Add(-21 * 24 * time.Hour).Format("2006-01-02T15:04:05Z")
	tasks := make([]Task, 0)

	offset := 0
	for {
		url := "/task" +
			"?limit=20" +
			"&offset=" + strconv.Itoa(offset) +
			"&type=YA_PACKAGE_2" +
			"&desc_re=" + descRe +
			"&fields=id,description,results.info,time.created,input_parameters.publish_to" +
			"&created=" + fmt.Sprintf("%s..%s", from, to)

		req, err := c.httpClient.NewGetRequest(ctx, url)
		if err != nil {
			return nil, err
		}

		var resp struct {
			Items []struct {
				ID          int64  `json:"id"`
				Description string `json:"description"`
				Created     string `json:"time.created"`
				Result      string `json:"results.info"`
				Repository  string `json:"input_parameters.publish_to"`
			} `json:"items"`
			Total int `json:"total"`
		}

		err = c.httpClient.SendJSONRequest(req, &resp)
		if err != nil {
			return nil, err
		}

		pkgRe := regexp.MustCompile("[a-z0-9-]+=[0-9]+\\." + branchName)
		tz, err := time.LoadLocation("Europe/Moscow")
		if err != nil {
			return nil, err
		}

		for _, item := range resp.Items {
			log.Println(item)
			if pkgs := pkgRe.FindAllString(item.Result, -1); len(pkgs) != 0 {
				created, err := time.Parse("2006-01-02T15:04:05.999999Z", item.Created)
				if err != nil {
					created = time.Time{}
				}
				createdStr := created.In(tz).Format("02 Jan 15:04")
				task := Task{
					Created:    created,
					CreatedStr: createdStr,
					Title:      item.Description,
					ID:         item.ID,
					Packages:   pkgs,
					Repository: item.Repository,
				}

				tasks = append(tasks, task)
			}
		}

		offset += 20
		if offset > resp.Total {
			break
		}
	}

	sort.Slice(tasks, func(i, j int) bool {
		return tasks[i].Created.Before(tasks[j].Created)
	})

	return tasks, nil
}

func (c *Client) CreateYaPackage(ctx context.Context, p *PackageParameters) (int, error) {
	taskKeyRobot := "robot-solomon-build"

	if p.MemoryGb == 0 {
		p.MemoryGb = 4
	}
	if p.DiskGb == 0 {
		p.DiskGb = 50
	}
	if p.Timeout == 0 {
		p.Timeout = time.Hour
	}
	if p.DebianDistribution == "" {
		p.DebianDistribution = "unstable"
	}
	if p.DuploadMaxAttempts == 0 {
		p.DuploadMaxAttempts = 3
	}

	arcadiaURL := "arcadia:/arc/" + p.Branch + "/arcadia"
	version := p.Branch[strings.LastIndexByte(p.Branch, '/')+1:]
	if p.Revision != "" {
		arcadiaURL += "@" + p.Revision
		version = p.Revision + "." + version
	} else {
		version = "<latest>." + version
	}
	desc := "SOLOMON"
	for _, s := range p.PkgPaths {
		s = s[:strings.LastIndexByte(s, '/')]
		desc += " " + s[strings.LastIndexByte(s, '/')+1:] + "=" + version
	}
	task := &TaskParameters{
		Owner:          "SOLOMON",
		Type:           "YA_PACKAGE_2",
		Description:    desc,
		MaxRestarts:    3,
		KillTimeout:    int(p.Timeout.Seconds()),
		FailOnAnyError: true,
	}
	task.Priority.Class = "SERVICE"
	task.Priority.Subclass = "HIGH"
	task.Requirements.ClientTags = p.Platform
	task.Requirements.Platform = "linux"
	task.Requirements.RAM = p.MemoryGb * (1 << 30)
	task.Requirements.DiskSpace = p.DiskGb * (1 << 30)
	task.CustomFields = []*CustomField{
		{"checkout", true},
		{"checkout_arcadia_from_url", arcadiaURL},
		{"use_aapi_fuse", true},
		{"aapi_fallback", true},
		{"build_type", "release"},
		{"packages", strings.Join(p.PkgPaths, ";")},
		{"package_type", "debian"},
		{"debian_distribution", p.DebianDistribution},
		{"dupload_max_attempts", p.DuploadMaxAttempts},
		{"full_strip_binaries", p.FullStrip},
		{"strip_binaries", p.Strip},
		{"changelog", p.Changelog},
		{"key_user", taskKeyRobot},
		{"binary_executor_release_type", "none"},
		{"ya_yt_token_yav_secret", p.YtStoreToken},
	}
	if len(p.Repos) == 1 {
		task.CustomFields = append(task.CustomFields, []*CustomField{
			{"publish_package", true},
			{"publish_to", p.Repos[0]},
		}...)

	} else if len(p.Repos) > 1 {
		task.CustomFields = append(task.CustomFields, []*CustomField{
			{"multiple_publish", true},
			{"multiple_publish_to", strings.Join(p.Repos, ";")},
		}...)
	}

	req, err := c.httpClient.NewPostJSONRequest(ctx, "/task", &task)
	if err != nil {
		return 0, fmt.Errorf("sandbox task failed to create schedule request, %v", err)
	}
	var resp struct {
		ID     int    `json:"id"`
		Status string `json:"status"`
	}
	err = c.httpClient.SendJSONRequest(req, &resp)
	if err != nil {
		return 0, fmt.Errorf("sandbox task failed to schedule, %v", err)
	}
	if resp.Status != "DRAFT" {
		return resp.ID, fmt.Errorf("bad status for newly created task, %s", resp.Status)
	}
	return resp.ID, nil
}

func (c *Client) StartTask(ctx context.Context, id int) error {
	start := struct {
		Comment string `json:"comment"`
		ID      []int  `json:"id"`
	}{
		ID: []int{id},
	}
	req, err := c.httpClient.NewPutJSONRequest(ctx, "/batch/tasks/start", &start)
	if err != nil {
		return fmt.Errorf("sandbox task failed to create start request, %v", err)
	}
	var resp []struct {
		ID      int    `json:"id"`
		Message string `json:"message"`
		Status  string `json:"status"`
	}
	// Somehow while sending request body we could write it without recognizing it here:
	// https://cs.opensource.google/go/go/+/refs/tags/go1.17.7:src/net/http/transfer.go;l=368
	// We can retry starting, in case of duplicate request we should get status=WARNING
	// https://st.yandex-team.ru/SOLOMON-8238
	for try := 0; try < taskStartTries; try++ {
		if err = c.httpClient.SendJSONRequest(req, &resp); err != nil {
			err = fmt.Errorf("sandbox task failed to start on %d try, %v", try, err)
		} else if len(resp) == 1 {
			if resp[0].Status == "SUCCESS" || resp[0].Status == "WARNING" {
				return nil
			}
			err = fmt.Errorf("bad task start status, %s", resp[0].Status)
		} else {
			err = fmt.Errorf("bad task start response, %v", resp)
		}
		time.Sleep(retryDelay)
	}
	return err
}

func (c *Client) GetTaskStatus(ctx context.Context, id int) (TaskStatus, error) {
	req, err := c.httpClient.NewGetRequest(ctx, "/task/"+strconv.Itoa(id))
	if err != nil {
		return Unknown, fmt.Errorf("sandbox task failed to create wait request, %v", err)
	}

	var resp struct {
		ID     int    `json:"id"`
		Status string `json:"status"`
	}
	err = c.httpClient.SendJSONRequest(req, &resp)
	if err != nil {
		return Unknown, fmt.Errorf("sandbox task failed to get response, %v", err)
	}
	status, ok := NameTaskStatuses[resp.Status]
	if !ok {
		return Unknown, fmt.Errorf("sandbox task bad status, %v", resp.Status)
	}
	return status, nil
}

func (c *Client) GetTaskResources(ctx context.Context, id int) (*Task, error) {
	req, err := c.httpClient.NewGetRequest(ctx, "/resource"+
		"?task_id="+strconv.Itoa(id)+
		"&state=READY"+
		"&limit=1024")
	if err != nil {
		return nil, fmt.Errorf("sandbox task failed to create get pkgs request, %v", err)
	}

	var resp struct {
		Items []struct {
			Attributes map[string]interface{} `json:"attributes"`
			Filename   string                 `json:"file_name"`
			Type       string                 `json:"type"`
			URL        string                 `json:"url"`
		} `json:"items"`
	}
	err = c.httpClient.SendJSONRequest(req, &resp)
	if err != nil {
		return nil, fmt.Errorf("sandbox task failed to get response, %v", err)
	}

	task := &Task{
		ID: int64(id),
	}
	for _, item := range resp.Items {
		if name, ok := item.Attributes["resource_name"]; ok {
			if version, ok := item.Attributes["resource_version"]; ok {
				ns, ok := name.(string)
				if !ok {
					continue
				}
				vs, ok := version.(string)
				if !ok {
					continue
				}
				task.Packages = append(task.Packages, ns+"="+vs)
			}
		}
	}
	return task, nil
}
