package sandbox

import "time"

// from https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/common/types/task.py?rev=r8088426#L174

type TaskStatus int

const (
	Unknown TaskStatus = iota
	// DRAFT:
	Draft
	// QUEUE:
	Enqueuing
	Enqueued
	// EXECUTE:
	Assigned
	Preparing
	Executing
	Suspending
	Susdended
	Resuming
	Temporary
	Finishing
	Stopping
	// WAIT:
	WaitRes
	WaitTask
	WaitTime
	WaitOut
	WaitMutex
	// FINISH:
	Success
	Releasing
	Released
	NotReleased
	Failure
	Deleted
	// BREAK:
	Exception
	NoRes
	Timeout
	Stopped
	Expired
)

var TaskStatusNames = []string{
	Unknown:     "UNKNOWN",
	Draft:       "DRAFT",
	Enqueuing:   "ENQUEUING",
	Enqueued:    "ENQUEUED",
	Assigned:    "ASSIGNED",
	Preparing:   "PREPARING",
	Executing:   "EXECUTING",
	Suspending:  "SUSPENDING",
	Susdended:   "SUSPENDED",
	Resuming:    "RESUMING",
	Temporary:   "TEMPORARY",
	Finishing:   "FINISHING",
	Stopping:    "STOPPING",
	WaitRes:     "WAIT_RES",
	WaitTask:    "WAIT_TASK",
	WaitTime:    "WAIT_TIME",
	WaitOut:     "WAIT_OUT",
	WaitMutex:   "WAIT_MUTEX",
	Success:     "SUCCESS",
	Releasing:   "RELEASING",
	Released:    "RELEASED",
	NotReleased: "NOT_RELEASED",
	Failure:     "FAILURE",
	Deleted:     "DELETED",
	Exception:   "EXCEPTION",
	NoRes:       "NO_RES",
	Timeout:     "TIMEOUT",
	Stopped:     "STOPPED",
	Expired:     "EXPIRED",
}

var NameTaskStatuses = map[string]TaskStatus{
	"UNKNOWN":      Unknown,
	"DRAFT":        Draft,
	"ENQUEUING":    Enqueuing,
	"ENQUEUED":     Enqueued,
	"ASSIGNED":     Assigned,
	"PREPARING":    Preparing,
	"EXECUTING":    Executing,
	"SUSPENDING":   Suspending,
	"SUSPENDED":    Susdended,
	"RESUMING":     Resuming,
	"TEMPORARY":    Temporary,
	"FINISHING":    Finishing,
	"STOPPING":     Stopping,
	"WAIT_RES":     WaitRes,
	"WAIT_TASK":    WaitTask,
	"WAIT_TIME":    WaitTime,
	"WAIT_OUT":     WaitOut,
	"WAIT_MUTEX":   WaitMutex,
	"SUCCESS":      Success,
	"RELEASING":    Releasing,
	"RELEASED":     Released,
	"NOT_RELEASED": NotReleased,
	"FAILURE":      Failure,
	"DELETED":      Deleted,
	"EXCEPTION":    Exception,
	"NO_RES":       NoRes,
	"TIMEOUT":      Timeout,
	"STOPPED":      Stopped,
	"EXPIRED":      Expired,
}

var FinishStatusSet = map[TaskStatus]struct{}{
	Success:     {},
	Releasing:   {},
	Released:    {},
	NotReleased: {},
	Failure:     {},
	Deleted:     {},
	Exception:   {},
	NoRes:       {},
	Timeout:     {},
	Stopped:     {},
	Expired:     {},
}

type Task struct {
	Created    time.Time
	CreatedStr string
	ID         int64
	Title      string
	Packages   []string
	Repository string
}

type PackageParameters struct {
	Branch             string
	Revision           string
	PkgPaths           []string
	Changelog          string
	Strip              bool
	FullStrip          bool
	Repos              []string
	DebianDistribution string
	DuploadMaxAttempts int
	Platform           string
	MemoryGb           int
	DiskGb             int
	Timeout            time.Duration
	YtStoreToken       string
}

type CustomField struct {
	Name  string      `json:"name"`
	Value interface{} `json:"value"`
}

type TaskParameters struct {
	Owner          string         `json:"owner"`
	Type           string         `json:"type"`
	Description    string         `json:"description,omitempty"`
	CustomFields   []*CustomField `json:"custom_fields,omitempty"`
	MaxRestarts    int            `json:"max_restarts,omitempty"`
	KillTimeout    int            `json:"kill_timeout,omitempty"`
	FailOnAnyError bool           `json:"fail_on_any_error,omitempty"`
	Priority       struct {
		Class    string `json:"class"`
		Subclass string `json:"subclass"`
	} `json:"priority"`
	Requirements struct {
		ClientTags string `json:"client_tags,omitempty"`
		Cores      int    `json:"cores,omitempty"`
		DiskSpace  int    `json:"disk_space,omitempty"`
		Platform   string `json:"platform,omitempty"`
		RAM        int    `json:"ram,omitempty"`
	} `json:"requirements"`
	Tags          []string `json:"tags,omitempty"`
	Notifications []struct {
		CheckStatus string   `json:"check_status,omitempty"`
		Recipients  []string `json:"recipients,omitempty"`
		Statuses    []string `json:"statuses,omitempty"`
		Transport   string   `json:"transport,omitempty"`
	} `json:"notifications,omitempty"`
}
