package svn

import (
	"context"
	"encoding/xml"
	"fmt"
	"os/exec"
	"path/filepath"
	"regexp"
	"strconv"
)

const (
	arcSvn          = "svn+ssh://arcadia.yandex.ru/arc"
	arcTrunk        = arcSvn + "/trunk"
	arcadiaTrunk    = arcSvn + "/trunk/arcadia"
	solomonBranches = arcSvn + "/branches/solomon"
)

func checkSvnPresent() error {
	if _, err := exec.LookPath("svn"); err != nil {
		return fmt.Errorf("svn command not found in your $PATH")
	}
	return nil
}

func formatCmdError(msg string, err error) error {
	if ee, ok := err.(*exec.ExitError); ok {
		return fmt.Errorf("%s (exit_code = %d)\n%s", msg, ee.ExitCode(), string(ee.Stderr))
	}
	return fmt.Errorf("%s: %w", msg, err)
}

func GetLatestRevision(ctx context.Context, branch, path string) (int, error) {
	if err := checkSvnPresent(); err != nil {
		return 0, err
	}

	cmd := exec.CommandContext(ctx, "svn", "log", "--limit", "1", "--xml", arcSvn+"/"+branch+"/arcadia"+path)
	out, err := cmd.Output()
	if err != nil {
		return 0, formatCmdError("svn log failed", err)
	}

	var l svnLog
	err = xml.Unmarshal(out, &l)
	if err != nil {
		return 0, fmt.Errorf("invalid svn log output: %w\n%s", err, string(out))
	}

	if len(l.Entries) == 0 {
		return 0, fmt.Errorf("empty svn log")
	}
	return l.Entries[0].Revision, nil
}

func ForkReleaseBranch(ctx context.Context, name string, revision int) error {
	if err := checkSvnPresent(); err != nil {
		return err
	}

	cmd := exec.CommandContext(ctx, "svn", "copy",
		"--revision", strconv.Itoa(revision),
		"--message", fmt.Sprintf("make new stable branch for solomon from r%d", revision),
		arcTrunk,
		solomonBranches+"/"+name)

	if err := cmd.Run(); err != nil {
		return formatCmdError("svn copy failed", err)
	}
	return nil
}

func ListReleaseBranches(ctx context.Context) ([]string, error) {
	if err := checkSvnPresent(); err != nil {
		return nil, err
	}

	cmd := exec.CommandContext(ctx, "svn", "ls", "--xml", solomonBranches)
	out, err := cmd.Output()
	if err != nil {
		return nil, formatCmdError("svn ls failed", err)
	}

	var list svnList
	err = xml.Unmarshal(out, &list)
	if err != nil {
		return nil, fmt.Errorf("invalid svn log output: %w", err)
	}

	names := make([]string, 0, 10)
	for _, l := range list.Lists {
		for _, e := range l.Entries {
			if e.Kind == "dir" {
				names = append(names, e.Name)
			}
		}
	}
	return names, nil
}

func RemoveReleaseBranch(ctx context.Context, name string) error {
	if err := checkSvnPresent(); err != nil {
		return err
	}

	cmd := exec.CommandContext(ctx, "svn", "rm", "--message", "remove outdated branch", solomonBranches+"/"+name)
	if err := cmd.Run(); err != nil {
		return formatCmdError("svn rm failed", err)
	}
	return nil
}

func CheckoutReleaseBranch(ctx context.Context, name, path string) error {
	if err := checkSvnPresent(); err != nil {
		return err
	}

	// checkout only immediates, to avoid checkout whole arcadia
	{
		cmd := exec.CommandContext(ctx, "svn", "checkout", "--depth", "immediates", solomonBranches+"/"+name)
		cmd.Dir = path
		if err := cmd.Run(); err != nil {
			return formatCmdError("svn checkout failed", err)
		}
	}

	// then checkout solomon dir and minimal stuff to be able to build packages from release branch
	{
		toUpdate := []string{
			".arcadia.root",
			"ya",
			"build",
			"library",
			"solomon",
		}

		cmd := exec.CommandContext(ctx, "svn", append([]string{"update", "--quiet", "--set-depth", "infinity"}, toUpdate...)...)
		cmd.Dir = filepath.Join(path, name, "arcadia")
		if err := cmd.Run(); err != nil {
			return formatCmdError("svn update failed", err)
		}
	}

	return nil
}

func Update(ctx context.Context, path string) error {
	if err := checkSvnPresent(); err != nil {
		return err
	}

	cmd := exec.CommandContext(ctx, "svn", "update")
	cmd.Dir = path

	if err := cmd.Run(); err != nil {
		return formatCmdError("svn update failed", err)
	}
	return nil
}

func MergeFromTrunk(ctx context.Context, path string, revisions []int) error {
	if err := checkSvnPresent(); err != nil {
		return err
	}

	for _, rev := range revisions {
		cmd := exec.CommandContext(ctx, "svn", "merge", "-r", fmt.Sprintf("%d:%d", rev-1, rev), arcTrunk+"/", ".")
		cmd.Dir = path
		if err := cmd.Run(); err != nil {
			return formatCmdError(fmt.Sprintf("svn merge failed for revision %d", rev), err)
		}
	}

	return nil
}

func PrintStatus(ctx context.Context, path string) (string, error) {
	if err := checkSvnPresent(); err != nil {
		return "", err
	}

	cmd := exec.CommandContext(ctx, "svn", "status")
	cmd.Dir = path

	if out, err := cmd.CombinedOutput(); err != nil {
		return "", formatCmdError("svn status failed", err)
	} else {
		return string(out), nil
	}
}

func Commit(ctx context.Context, path, message string) (int, error) {
	if err := checkSvnPresent(); err != nil {
		return 0, err
	}

	cmd := exec.CommandContext(ctx, "svn", "commit", "-m", message)
	cmd.Dir = path

	if out, err := cmd.Output(); err != nil {
		return 0, formatCmdError("svn commit failed", err)
	} else {
		re := regexp.MustCompile(`Committed revision (\d+)`)
		match := re.FindSubmatch(out)
		if len(match) < 2 {
			return 0, fmt.Errorf("cannot parse revision number from svn commit output")
		}
		return strconv.Atoi(string(match[1]))
	}
}
