package z2

import (
	"a.yandex-team.ru/solomon/libs/go/uhttp"
	"a.yandex-team.ru/solomon/tools/release/internal/apt"
	"context"
	"encoding/json"
	"fmt"
	"time"
)

const (
	internalBaseURL = "https://z2.yandex-team.ru"
	cloudBaseURL    = "https://z2-cloud.yandex-team.ru"
	httpTimeout     = 20 * time.Second
)

type Client interface {
	ListItems(ctx context.Context, configID string) (apt.PackageList, error)
	EditItems(ctx context.Context, configID string, items apt.PackageList) error
	Update(ctx context.Context, configID string) error
	UpdateStatus(ctx context.Context, configID string) (*UpdateStatus, error)
	Workers(ctx context.Context, configID string) (*Workers, error)
	ControlPanelURL(configID string) string
}

func NewClient(apiKeys APIKeys, isCloud bool) Client {
	baseURL := internalBaseURL
	if isCloud {
		baseURL = cloudBaseURL
	}
	return &clientImpl{
		httpClient: uhttp.NewClient(baseURL+"/api/v1", "solomon-release-tool", nil, httpTimeout, true),
		apiKeys:    apiKeys,
		baseURL:    baseURL,
	}
}

type clientImpl struct {
	httpClient *uhttp.Client
	apiKeys    APIKeys
	baseURL    string
}

func (c *clientImpl) ListItems(ctx context.Context, configID string) (apt.PackageList, error) {
	apiKey, ok := c.apiKeys[configID]
	if !ok {
		return nil, fmt.Errorf("unknown Z2 API key for configuration %s. Please update z2_api_keys secret in YAV", configID)
	}

	req, err := c.httpClient.NewGetRequest(ctx, "/items?"+uhttp.Query(
		"configId", configID,
		"apiKey", apiKey,
	).String())
	if err != nil {
		return nil, err
	}
	req.Header.Add("Accept", "application/json")

	var resp struct {
		Success  bool   `json:"success"`
		ErrorMsg string `json:"errorMsg"`
		Response struct {
			Items apt.PackageList `json:"items"`
		} `json:"response"`
	}
	if err = c.httpClient.SendJSONRequest(req, &resp); err != nil {
		return nil, err
	}
	if !resp.Success {
		return nil, fmt.Errorf("bad response: %s", resp.ErrorMsg)
	}
	return resp.Response.Items, nil
}

func (c *clientImpl) EditItems(ctx context.Context, configID string, items apt.PackageList) error {
	apiKey, ok := c.apiKeys[configID]
	if !ok {
		return fmt.Errorf("unknown Z2 API key for configuration %s. Please update z2_api_keys secret in YAV", configID)
	}

	itemsBytes, err := json.Marshal(items)
	if err != nil {
		return fmt.Errorf("cannot marshal items to json: %w", err)
	}

	req, err := c.httpClient.NewPostRequest(ctx, "/editItems", uhttp.Query(
		"configId", configID,
		"apiKey", apiKey,
		"items", itemsBytes,
	).Bytes())
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")

	_, err = c.httpClient.SendRequest(req)
	return err
}

func (c *clientImpl) Update(ctx context.Context, configID string) error {
	apiKey, ok := c.apiKeys[configID]
	if !ok {
		return fmt.Errorf("unknown Z2 API key for configuration %s. Please update z2_api_keys secret in YAV", configID)
	}

	req, err := c.httpClient.NewPostRequest(ctx, "/update", uhttp.Query(
		"configId", configID,
		"apiKey", apiKey,
	).Bytes())
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")

	_, err = c.httpClient.SendRequest(req)
	return err
}

func (c *clientImpl) UpdateStatus(ctx context.Context, configID string) (*UpdateStatus, error) {
	apiKey, ok := c.apiKeys[configID]
	if !ok {
		return nil, fmt.Errorf("unknown Z2 API key for configuration %s. Please update z2_api_keys secret in YAV", configID)
	}

	req, err := c.httpClient.NewGetRequest(ctx, "/updateStatus?"+uhttp.Query(
		"configId", configID,
		"apiKey", apiKey,
	).String())
	if err != nil {
		return nil, err
	}
	req.Header.Add("Accept", "application/json")

	var resp struct {
		Success  bool         `json:"success"`
		ErrorMsg string       `json:"errorMsg"`
		Response UpdateStatus `json:"response"`
	}
	if err = c.httpClient.SendJSONRequest(req, &resp); err != nil {
		return nil, err
	}

	if !resp.Success {
		return nil, fmt.Errorf("non success response: %s", resp.ErrorMsg)
	}

	return &resp.Response, err
}

func (c *clientImpl) Workers(ctx context.Context, configID string) (*Workers, error) {
	apiKey, ok := c.apiKeys[configID]
	if !ok {
		return nil, fmt.Errorf("unknown Z2 API key for configuration %s. Please update z2_api_keys secret in YAV", configID)
	}

	req, err := c.httpClient.NewGetRequest(ctx, "/workers?"+uhttp.Query(
		"configId", configID,
		"apiKey", apiKey,
	).String())
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	req.Header.Add("Accept", "application/json")

	var resp struct {
		Success  bool    `json:"success"`
		ErrorMsg string  `json:"errorMsg"`
		Response Workers `json:"response"`
	}
	if err = c.httpClient.SendJSONRequest(req, &resp); err != nil {
		return nil, err
	}

	if !resp.Success {
		return nil, fmt.Errorf("non success response: %s", resp.ErrorMsg)
	}

	return &resp.Response, err
}

func (c *clientImpl) ControlPanelURL(configID string) string {
	return c.baseURL + "/control_panel?configId=" + configID
}
