package z2

import (
	"context"
	"fmt"
	"log"
	"strings"
	"time"

	"a.yandex-team.ru/solomon/tools/release/internal/apt"
)

// ==========================================================================================

type Worker struct {
	Cfg    string
	client Client
}

func NewZW(apiKeys APIKeys, cfg string) *Worker {
	return &Worker{
		client: NewClient(apiKeys, strings.Contains(cfg, "_CLOUD_")),
		Cfg:    cfg,
	}
}

func (w *Worker) SelectiveEdit(pkgs apt.PackageList) (int, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 20*time.Second)
	defer cancel()

	pkgCurrent, err := w.client.ListItems(ctx, w.Cfg)
	if err != nil {
		return 0, err
	}
	pkgUpdate := make(apt.PackageList, 0)
	for _, pc := range pkgCurrent {
		for _, ps := range pkgs {
			if pc.Name != ps.Name || ps.Version == nil {
				continue
			}
			vComp := ps.Version.Compare(pc.Version)
			if vComp != 0 {
				pkgUpdate = append(pkgUpdate, ps)
				if vComp < 0 {
					log.Printf("Package %s at %s will be downgraded %s -> %s", ps.Name, w.Cfg, pc.Version.String(), ps.Version.String())
				}
			}
		}
	}
	if len(pkgUpdate) == 0 {
		return 0, nil
	}
	return len(pkgUpdate), w.client.EditItems(ctx, w.Cfg, pkgUpdate)
}

func (w *Worker) UpdateStart() error {
	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	return w.client.Update(ctx, w.Cfg)
}

func (w *Worker) UpdateWait() error {
	ctx, cancel := context.WithTimeout(context.Background(), 15*time.Minute)
	defer cancel()

	z2URL := w.client.ControlPanelURL(w.Cfg)
	for {
		status, err := w.client.UpdateStatus(ctx, w.Cfg)
		if err != nil {
			return err
		}
		if status.IsFinished() {
			if status.Result == "SUCCESS" {
				log.Printf("Success updating %s", z2URL)
				return nil
			} else {
				log.Printf("Done %s: %s - failed workers: %v", z2URL, status.Result, status.FailedWorkers)
				return fmt.Errorf("failed with status %s", status.Result)
			}
		}
		log.Printf("    %s", status.UpdateStatus)
		time.Sleep(10 * time.Second)
	}
}

func (w *Worker) RefURL() string {
	return w.client.ControlPanelURL(w.Cfg)
}
