package cmd

import (
	"a.yandex-team.ru/library/go/yandex/oauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/solomon/protos/secrets"
	"bufio"
	"context"
	"fmt"
	"github.com/golang/protobuf/proto"
	"github.com/spf13/cobra"
	"os"
	"regexp"
)

const (
	clientID     = "e397b846943844228aca41877a8b85d6"
	clientSecret = "aa15736d8bc8492c8bfa0578d9892f3f"
)

var (
	config    string
	yavRegexp = regexp.MustCompile("\"?yav://(sec-[a-z0-9]+)/([^\"]+)\"?")
)

func init() {
	createCmd := &cobra.Command{
		Use:   "get",
		Short: "Get secrets from yav.yandex-team.ru used in configuration file",
		RunE:  runCreateCmd,
	}
	createCmd.Flags().StringVar(&config, "config", "", "path to a configuration file to parse")
	_ = createCmd.MarkFlagRequired("config")
	rootCmd.AddCommand(createCmd)
}

type secretRef struct {
	id, key string
}

func (ref *secretRef) Format() string {
	return fmt.Sprintf("%s/%s", ref.id, ref.key)
}

func parseKey(s string) *secretRef {
	keyGroups := yavRegexp.FindStringSubmatch(s)
	if keyGroups == nil {
		return nil
	}

	return &secretRef{keyGroups[1], keyGroups[2]}
}

func findSecretRefs(filename string) ([]secretRef, error) {
	fmt.Fprintln(os.Stderr, "parsing "+filename)

	file, err := os.Open(filename)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	var secretRefs []secretRef

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		if key := parseKey(scanner.Text()); key != nil {
			secretRefs = append(secretRefs, *key)
			fmt.Fprintf(os.Stderr, "found secret reference {id: %s, key: %s}\n", key.id, key.key)
		}
	}

	if err := scanner.Err(); err != nil {
		return nil, err
	}

	return secretRefs, nil
}

func loadSecrets(ctx context.Context, secretRefs []secretRef) (map[string]string, error) {
	token, err := oauth.GetTokenBySSH(ctx, clientID, clientSecret)
	if err != nil {
		return nil, fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}

	client, err := httpyav.NewClient(httpyav.WithOAuthToken(token))
	if err != nil {
		return nil, fmt.Errorf("cannot initialize yav client, %v", err)
	}

	var secretsMap = make(map[string]string)

	for _, ref := range secretRefs {
		key := ref.Format()
		if _, ok := secretsMap[key]; ok {
			continue
		}

		ver, err := client.GetVersion(ctx, ref.id)
		if err != nil {
			return nil, fmt.Errorf("cannot load sercret %s, %v", ref.id, err)
		}

		var secretValue *string = nil
		for _, value := range ver.Version.Values {
			if value.Key == ref.key {
				secretValue = &value.Value
				break
			}
		}

		if secretValue == nil {
			return nil, fmt.Errorf("secret %s does not have key %s", ref.id, ref.key)
		}

		secretsMap[key] = *secretValue
	}

	return secretsMap, nil
}

func serializeSecrets(secretsMap map[string]string) error {
	var protoSecrets []*secrets.Secret
	for key, value := range secretsMap {
		protoSecrets = append(protoSecrets, &secrets.Secret{Key: key, Value: value})
	}

	if err := proto.MarshalText(os.Stdout, &secrets.SecretsConfig{Secrets: protoSecrets}); err != nil {
		return fmt.Errorf("cannot serialize secrets as text protobuf, %v", err)
	}
	return nil
}

func runCreateCmd(cmd *cobra.Command, args []string) error {
	secretRefs, err := findSecretRefs(config)
	if err != nil {
		return fmt.Errorf("cannot read file %s, %v", config, err)
	}

	secretsMap, err := loadSecrets(context.Background(), secretRefs)
	if err != nil {
		return err
	}

	return serializeSecrets(secretsMap)
}
