package kms

import (
	kmsApi "a.yandex-team.ru/cloud/bitbucket/private-api/yandex/cloud/priv/kms/v1"
	grpcSolomon "a.yandex-team.ru/solomon/libs/go/grpc"
	"a.yandex-team.ru/solomon/protos/secrets"
	"context"
	"fmt"
	"google.golang.org/grpc"
	"io"
)

var kmsAddress = map[secrets.CloudEnv]string{
	secrets.CloudEnv_PREPROD: "kms.cloud-preprod.yandex.net:8443",
	secrets.CloudEnv_PROD:    "kms.yandex:8443",
	secrets.CloudEnv_GPN:     "kms.gpn.yandexcloud.net:8443",
}

type Client interface {
	io.Closer

	Encrypt(ctx context.Context, keyID string, plaintext []byte) ([]byte, error)
	Decrypt(ctx context.Context, keyID string, ciphertext []byte) ([]byte, error)
}

func NewClient(ctx context.Context, iamToken string, env secrets.CloudEnv) (Client, error) {
	addr := kmsAddress[env]
	conn, err := grpcSolomon.NewConnection(ctx, addr, grpcSolomon.NewIamTokenCredentials(iamToken))
	if err != nil {
		return nil, fmt.Errorf("cannot connect to %s, %v", addr, err)
	}
	return &kmsClient{conn, kmsApi.NewSymmetricCryptoServiceClient(conn)}, nil
}

type kmsClient struct {
	conn   *grpc.ClientConn
	client kmsApi.SymmetricCryptoServiceClient
}

func (c *kmsClient) Close() error {
	return c.conn.Close()
}

func (c *kmsClient) Encrypt(ctx context.Context, keyID string, plaintext []byte) ([]byte, error) {
	request := &kmsApi.SymmetricEncryptRequest{
		KeyId:     keyID,
		Plaintext: plaintext}

	response, err := c.client.Encrypt(ctx, request)
	if err != nil {
		return nil, fmt.Errorf("cannot encrypt plaintext with KMS key %s, %v", keyID, err)
	}

	return response.Ciphertext, nil
}

func (c *kmsClient) Decrypt(ctx context.Context, keyID string, ciphertext []byte) ([]byte, error) {
	request := &kmsApi.SymmetricDecryptRequest{
		KeyId:      keyID,
		Ciphertext: ciphertext}

	response, err := c.client.Decrypt(ctx, request)
	if err != nil {
		return nil, fmt.Errorf("cannot decrypt ciphertext with KMS key %s, %v", keyID, err)
	}

	return response.Plaintext, nil
}
