package cmd

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/tracker"
	"a.yandex-team.ru/solomon/tools/service-dashboard/internal/client"
	"a.yandex-team.ru/solomon/tools/service-dashboard/internal/config"
	"a.yandex-team.ru/solomon/tools/service-dashboard/internal/utils"
	"context"
	"encoding/json"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"time"
)

var skipFields = map[string]bool{
	"createdAt":           true,
	"createdBy":           true,
	"id":                  true,
	"updatedAt":           true,
	"updatedBy":           true,
	"version":             true,
	"sourceDashboardLink": true,
}

func init() {
	serviceDashboardCmd := &cobra.Command{
		Use:   "copy [flags] <from> <to>",
		Short: "Copy service dashboard from one ENV to another. ",
		Long: "Copy service dashboard from cloud_pre (https://solomon.cloud-preprod.yandex-team.ru) to cloud_prod (https://solomon.cloud.yandex-team.ru) " +
			"by default. Dashboard from cloud_prod would be pasted to https://paste.yandex-team.ru/.\n" +
			"Where <from> and <to> can be:\n" +
			"   1) dashboard ids {{id}}\n" +
			"    Example:\n" +
			"          $ service-dashboard copy biiheb7ibas09sks01d6 fbeo9mfuup35slorffvu\n" +
			"  \n" +
			"   2) fully qualified dashboard id: {{env}}://{{id}}\n" +
			"    Example:\n" +
			"          $ service-dashboard copy cloud_pre://biiheb7ibas09sks01d6 cloud_prod://fbeo9mfuup35slorffvu\n" +
			"Available env: cloud_pre, cloud_prod, test, pre, prod",
		Args: func(cmd *cobra.Command, args []string) error {
			if len(args) != 2 {
				return fmt.Errorf("Usage: " + cmd.Use)
			}
			return nil
		},
		RunE: copyDashboard,
	}
	flags := serviceDashboardCmd.Flags()
	// optional flags
	flags.Bool("confirm", false, "auto confirm dashboard diff and update")
	flags.Bool("dry-run", false, "test execution without update dashboard")
	flags.String("ticket", "", "ticket to close")
	rootCmd.AddCommand(serviceDashboardCmd)
}

func copyDashboard(cmd *cobra.Command, args []string) error {
	config, err := config.ParseConfig(args)
	if err != nil {
		return err
	}

	confirm, _ := cmd.Flags().GetBool("confirm")
	dryRun, _ := cmd.Flags().GetBool("dry-run")
	ticket, _ := cmd.Flags().GetString("ticket")

	ctx := context.Background()

	log.Printf("Start copy tool :\nfrom api URL: %s\nto api URL: %s",
		formatInternalURL(config.FromAPIURL, config.FromDashboardID), formatInternalURL(config.ToAPIURL, config.ToDashboardID))

	fromClient := client.NewServiceDashboardClient(config.FromAPIURL, config.TokenOauth, config.TokenIAMFrom)
	toClient := client.NewServiceDashboardClient(config.ToAPIURL, config.TokenOauth, config.TokenIAMTo)
	pasteClient := client.NewPasteClient()

	fromDashboard, err := fromClient.GetDashboard(ctx, config.FromDashboardID)
	if err != nil {
		return err
	}
	toDashboard, err := toClient.GetDashboard(ctx, config.ToDashboardID)
	if err != nil {
		return err
	}

	sourceURL := formatUIURL(config.FromAPIURL, fromDashboard)
	log.Printf("From UI URL: %s", sourceURL)
	log.Printf("To UI URL: %s", formatUIURL(config.ToAPIURL, toDashboard))

	toDashboardBytes, err := json.MarshalIndent(toDashboard, "", " ")
	if err != nil {
		return err
	}

	preparedTo, err := prepareDashboardCopy(toDashboard)
	if err != nil {
		return err
	}
	preparedFrom, err := prepareDashboardCopy(fromDashboard)
	if err != nil {
		return err
	}

	err = utils.PrintDiff(preparedTo, preparedFrom)
	if err != nil {
		return err
	}
	if dryRun {
		log.Println(color.Red("Dashboard IS NOT copied!"))
		return nil
	}
	if !confirm {
		if !enterYN() {
			log.Println(color.Red("Dashboard IS NOT copied!"))
			return nil
		}
	}

	backupURL, err := pasteClient.Paste(ctx, string(toDashboardBytes))
	if err != nil {
		return err
	}
	log.Printf("Dashboard %s previous version saved here: %s", color.Blue(config.ToDashboardID), backupURL)

	fromDashboard["version"] = toDashboard["version"]
	fromDashboard["id"] = toDashboard["id"]
	fromDashboard["sourceDashboardLink"] = sourceURL

	fromDashboardBytes, err := json.Marshal(fromDashboard)
	if err != nil {
		return err
	}
	_, err = toClient.UpdateDashboard(ctx, config.ToDashboardID, string(fromDashboardBytes))
	if err != nil {
		return err
	}

	log.Println("Dashboard copied!")

	if len(ticket) > 0 {
		if err = closeTicket(ticket, config.TokenOauth); err != nil {
			return err
		}
		log.Printf("ticket %s closed", ticket)
	}

	return nil
}

func closeTicket(ticket string, oauth string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	trackerClient := tracker.NewClient(oauth)
	if err := trackerClient.ResolveIssue(ctx, ticket); err != nil {
		return fmt.Errorf("cannot close ticket %s: %w", ticket, err)
	}
	return nil
}

func prepareDashboardCopy(dashboardOriginal map[string]interface{}) ([]byte, error) {
	dashboard := make(map[string]interface{})
	for k, v := range dashboardOriginal {
		_, prs := skipFields[k]
		if prs {
			continue
		}
		dashboard[k] = v
	}
	bytes, err := json.MarshalIndent(dashboard, "", " ")
	if err != nil {
		return nil, err
	}
	return bytes, nil
}

func formatUIURL(APIURL string, dashboard map[string]interface{}) string {
	return fmt.Sprintf("%s/admin/serviceProviders/%s/serviceDashboards/%s", APIURL, dashboard["service"], dashboard["id"])
}

func formatInternalURL(APIURL string, dashboardID string) string {
	return fmt.Sprintf("%s/api/internal/serviceDashboards/%s", APIURL, dashboardID)
}

func enterYN() bool {
	for {
		log.Println("Update dashboard? y/n")
		var v string
		_, err := fmt.Scanln(&v)
		if err != nil {
			continue
		}
		if v == "y" {
			return true
		}
		if v == "n" {
			return false
		}
	}
}
