package config

import (
	"a.yandex-team.ru/library/go/yandex/oauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/solomon/libs/go/iam"
	"a.yandex-team.ru/solomon/protos/secrets"
	"context"
	"fmt"
	"log"
	"os"
	"os/user"
	"strings"
	"time"
)

const (
	cloudPreEnv        = "cloud_pre"
	cloudPreprodAPIURL = "https://solomon.cloud-preprod.yandex-team.ru"
	cloudProdEnv       = "cloud_prod"
	cloudProdAPIURL    = "https://solomon.cloud.yandex-team.ru"
	testEnv            = "test"
	testAPIURL         = "https://solomon-test.yandex-team.ru"
	preEnv             = "pre"
	preprodAPIURL      = "https://solomon-pre.yandex-team.ru"
	prodEnv            = "prod"
	prodAPIURL         = "https://solomon.yandex-team.ru"
	clientID           = "350dd730a52a4a018323255d235889f5"
	clientSecret       = "0dd1e4d7b381494180255114e2faa990"
)

type Config struct {
	TokenIAMFrom    string
	TokenIAMTo      string
	TokenOauth      string
	FromAPIURL      string
	ToAPIURL        string
	FromDashboardID string
	ToDashboardID   string
}

func ParseConfig(args []string) (*Config, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 20*time.Second)
	defer cancel()

	config := &Config{
		FromAPIURL: cloudPreprodAPIURL,
		ToAPIURL:   cloudProdAPIURL,
	}
	tokenOauth, err := getOauthToken(ctx)
	if err != nil {
		return nil, err
	}
	config.TokenOauth = tokenOauth

	envFrom := secrets.CloudEnv_value["PREPROD"]
	envTo := secrets.CloudEnv_value["PROD"]
	from := args[0]
	to := args[1]

	err = parseParam(from, &config.FromDashboardID, &config.FromAPIURL, &envFrom)
	if err != nil {
		return nil, err
	}
	err = parseParam(to, &config.ToDashboardID, &config.ToAPIURL, &envTo)
	if err != nil {
		return nil, err
	}
	t, err := getIamToken(ctx, envFrom, "IAM_SECRET_FROM")
	if err != nil && config.TokenOauth == "" {
		return nil, err
	} else if err != nil {
		log.Println("Oauth would be used in source client")
	}
	config.TokenIAMFrom = t

	t, err = getIamToken(ctx, envFrom, "IAM_SECRET_TO")
	if err != nil && config.TokenOauth == "" {
		return nil, err
	} else if err != nil {
		log.Println("Oauth would be used in target client")
	}
	config.TokenIAMTo = t

	return config, nil
}

func getIamToken(ctx context.Context, env int32, secretVariable string) (string, error) {
	keyJSON, err := loadYavKey(ctx, os.Getenv(secretVariable))
	if err != nil {
		return "", err
	}
	token, err := iam.GetIamToken(secrets.CloudEnv(env), keyJSON)
	if err != nil {
		return "", err
	}
	return token, nil
}

func getOauthToken(ctx context.Context) (string, error) {
	tokenOauth := os.Getenv("OAUTH_TOKEN")
	var err error
	if tokenOauth == "" {
		tokenOauth, err = oauth.GetTokenBySSH(ctx, clientID, clientSecret)
		if err != nil {
			return "", err
		}
		log.Println("OAuth token received with API")
	}
	return tokenOauth, nil
}

func parseParam(val string, dashboardID *string, apiURL *string, env *int32) error {
	if strings.Contains(val, "://") {
		vals := strings.Split(val, "://")
		*dashboardID = vals[1]
		if vals[0] == cloudPreEnv {
			*apiURL = cloudPreprodAPIURL
			*env = secrets.CloudEnv_value["PREPROD"]
		} else if vals[0] == cloudProdEnv {
			*apiURL = cloudProdAPIURL
			*env = secrets.CloudEnv_value["PROD"]
		} else if vals[0] == preEnv {
			*apiURL = preprodAPIURL
			*env = 0
		} else if vals[0] == prodEnv {
			*apiURL = prodAPIURL
			*env = 0
		} else if vals[0] == testEnv {
			*apiURL = testAPIURL
			*env = 0
		} else {
			return fmt.Errorf("unknown env %s", vals[0])
		}
	} else {
		*dashboardID = val
	}
	return nil
}

//below code is copied from "a.yandex-team.ru/solomon/tools/secrets"
func currentLogin() (string, error) {
	sysUser, err := user.Current()
	if err != nil {
		return "", err
	}
	return sysUser.Username, nil
}

func loadYavKey(ctx context.Context, secretKey string) (string, error) {
	secretKeyArr := strings.Split(secretKey, "/")
	if len(secretKeyArr) != 2 {
		return "", fmt.Errorf("invalid format of secret key: %s", secretKey)
	}

	login, err := currentLogin()
	if err != nil {
		return "", fmt.Errorf("cannot get login of current user, %v", err)
	}

	token, err := oauth.GetTokenBySSH(ctx, clientID, clientSecret, oauth.WithUserLogin(login))
	if err != nil {
		return "", fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}

	client, err := httpyav.NewClient(httpyav.WithOAuthToken(token))
	if err != nil {
		return "", fmt.Errorf("cannot initialize yav client, %v", err)
	}

	ver, err := client.GetVersion(ctx, secretKeyArr[0])
	if err != nil {
		return "", fmt.Errorf("cannot load sercret %s, %v", secretKeyArr[0], err)
	}

	for _, value := range ver.Version.Values {
		if value.Key == secretKeyArr[1] {
			return value.Value, nil
		}
	}

	return "", fmt.Errorf("cannot find %s in secret %s", secretKeyArr[1], secretKeyArr[0])
}
