package main

import (
	ingestor "a.yandex-team.ru/solomon/services/ingestor/api"
	"bufio"
	"encoding/base64"
	"flag"
	"fmt"
	"github.com/golang/protobuf/proto"
	"io"
	"io/ioutil"
	"log"
	"os"
	"path"
)

var source = flag.String("source", "", "Source file with binary trace file for ingestor")
var target = flag.String("target", "ammo.json", "Target file for append ammo")

func init() {
	flag.Parse()
	if *source == "" {
		_, _ = os.Stderr.WriteString("missing required -source argument\n")
		os.Exit(1)
	}
	if *target == "" {
		_, _ = os.Stderr.WriteString("missing required -target argument\n")
		os.Exit(1)
	}
}

type AmmoWriter struct {
	file   *os.File
	writer *bufio.Writer
}

func (a AmmoWriter) WriteMessage(isPull bool, req proto.Message) {
	bytes, err := proto.Marshal(req)
	if err != nil {
		log.Panicln("Unable marshal ", proto.MarshalTextString(req))
	}
	encoded := base64.StdEncoding.EncodeToString(bytes)
	_, err = a.writer.WriteString(fmt.Sprintf("{\"pull\": %t, \"req\": \"%s\"}\n", isPull, encoded))
	if err != nil {
		log.Panicln("Unable write message", err)
	}
}

func (a AmmoWriter) Close() error {
	err := a.writer.Flush()
	if err != nil {
		log.Panicln(err)
	}
	return a.file.Close()
}

func CreateAmmoWriter() AmmoWriter {
	file, err := os.OpenFile(*target, os.O_RDWR|os.O_CREATE|os.O_TRUNC, 0666)
	if err != nil {
		log.Fatalln("Unable open target file", err)
	}

	writer := bufio.NewWriter(file)
	return AmmoWriter{file: file, writer: writer}
}

type TraceReader struct {
	buffer *proto.Buffer
}

func CreateTraceReader(file string) TraceReader {
	content, err := ioutil.ReadFile(file)
	if err != nil {
		panic(err)
	}
	return TraceReader{buffer: proto.NewBuffer(content)}
}

func (a TraceReader) ReadMessageType() (uint64, error) {
	return a.buffer.DecodeVarint()
}

func (a TraceReader) ReadMessage(pb proto.Message) {
	err := a.buffer.DecodeMessage(pb)
	if err != nil {
		panic(err)
	}
}

func IsEndOfFile(err error) bool {
	return err == io.ErrUnexpectedEOF || err == io.EOF
}

func Convert(writer AmmoWriter, file string) {
	var reader = CreateTraceReader(file)
	pull := &ingestor.TPulledDataRequest{}
	push := &ingestor.TPushedDataRequest{}

	for {
		messageType, err := reader.ReadMessageType()
		if IsEndOfFile(err) {
			break
		} else if err != nil {
			panic(err)
		}

		switch messageType {
		case 0:
			reader.ReadMessage(pull)
			writer.WriteMessage(true, pull)
		case 1:
			reader.ReadMessage(push)
			writer.WriteMessage(false, push)
		default:
			log.Panicln("Unknorn message type:", messageType)
		}
	}
}

func main() {
	var writer = CreateAmmoWriter()
	defer writer.Close()

	stat, err := os.Stat(*source)
	if err != nil {
		log.Fatalln(err)
	}

	if stat.IsDir() {
		c, err := ioutil.ReadDir(*source)
		if err != nil {
			log.Fatalln(err)
		}

		for _, entry := range c {
			if entry.IsDir() {
				continue
			}

			Convert(writer, path.Join(*source, entry.Name()))
			log.Println("Success ", entry.Name())
		}
	} else {
		Convert(writer, *source)
	}
}
