package main

import (
	"context"
	"fmt"
	"github.com/yandex/pandora/core"
	"github.com/yandex/pandora/core/coreutil"
	"github.com/yandex/pandora/core/provider"
	"go.uber.org/zap"
	"io"
	"log"
)

type PreloadAmmoProviderConfig struct {
	Limit     int
	QueueSize int             `config:"ammo-queue-size" validate:"min=0"`
	Source    core.DataSource `config:"source" validate:"required"`
}

func NewPreloadAmmoProvider(conf PreloadAmmoProviderConfig) core.Provider {
	queueSize := conf.QueueSize
	if queueSize == 0 {
		queueSize = conf.Limit
	}
	if queueSize == 0 {
		queueSize = 256 * 1024
	}

	var p = &preloadAmmoProvider{
		pool: make([]core.Ammo, 0, conf.Limit),
		sink: make(chan core.Ammo, queueSize),
		conf: conf,
	}
	err := p.fillPool()
	if err != nil {
		panic(err)
	}
	return p
}

func DefaultPreloadAmmoProviderConfig() PreloadAmmoProviderConfig {
	return PreloadAmmoProviderConfig{}
}

type preloadAmmoProvider struct {
	idx  int
	pool []core.Ammo
	sink chan core.Ammo
	conf PreloadAmmoProviderConfig
}

func (p *preloadAmmoProvider) fillPool() error {
	source, err := p.conf.Source.OpenSource()
	if err != nil {
		return fmt.Errorf("data source open failed %w", err)
	}
	defer func() {
		err2 := source.Close()
		if err2 != nil {
			log.Println("data source close failed", err2)
		}
	}()

	decoder := WrapToProtoAmmoDecoder(provider.NewJSONAmmoDecoder(source, coreutil.DefaultBufferSize))
	unlimited := p.conf.Limit <= 0
	for p.idx = 0; unlimited || p.idx < p.conf.Limit; p.idx++ {
		ammo := new(ProtoAmmo)
		err = decoder.Decode(ammo)
		if err == io.EOF {
			log.Println("ProtoAmmo source finished, size ", p.idx)
			return nil
		}

		if err != nil {
			return fmt.Errorf("ammo #%v decode failed %w", p.idx, err)
		}

		p.pool = append(p.pool, ammo)
	}
	log.Println("ProtoAmmo limit is reached, decoded ", p.idx)
	return nil
}

func (p *preloadAmmoProvider) Run(ctx context.Context, deps core.ProviderDeps) error {
	defer close(p.sink)
	deps.Log.Info("Start ProtoAmmo provider", zap.Int("poolSize", len(p.pool)))
	for {
		for p.idx = 0; p.idx < len(p.pool); p.idx++ {
			select {
			case p.sink <- p.pool[p.idx]:
			case <-ctx.Done():
				return nil
			}
		}
	}
}

func (p *preloadAmmoProvider) Acquire() (a core.Ammo, ok bool) {
	a, ok = <-p.sink
	return
}

func (p *preloadAmmoProvider) Release(core.Ammo) {}
