package main

import (
	"a.yandex-team.ru/solomon/services/ingestor/api"
	"context"
	"fmt"
	"github.com/yandex/pandora/core"
	"log"
)

type IngestorClient interface {
	WritePushed(ctx context.Context, req *ingestor.TPushedDataRequest) (*ingestor.TDataProcessResponse, error)
	WritePulled(ctx context.Context, req *ingestor.TPulledDataRequest) (*ingestor.TDataProcessResponse, error)
	RefreshAssignments(ctx context.Context) error
}

type ingestorClientImpl struct {
	hosts       []HostClient
	hostByName  map[string]HostClient
	numIDByHost map[uint32]HostClient
	deps        core.GunDeps
}

func NewIngestorClient(ctx context.Context, target string) (IngestorClient, error) {
	hosts, port, err := ResolveTarget(target)
	if err != nil {
		return nil, err
	}

	clients := make([]HostClient, len(hosts))
	clientByHost := make(map[string]HostClient)
	clientByNumID := make(map[uint32]HostClient)
	for i, h := range hosts {
		client, err := NewHostClient(ctx, h, port)
		if err != nil {
			return nil, err
		}
		clients[i] = client
		clientByHost[h] = client
	}
	return &ingestorClientImpl{hosts: clients, hostByName: clientByHost, numIDByHost: clientByNumID}, nil
}

func (c *ingestorClientImpl) WritePushed(ctx context.Context, req *ingestor.TPushedDataRequest) (*ingestor.TDataProcessResponse, error) {
	target := c.numIDByHost[req.NumId]
	if target == nil {
		return nil, fmt.Errorf("unknown host for numId %d", req.NumId)
	}
	return target.WritePushed(ctx, req)
}

func (c *ingestorClientImpl) WritePulled(ctx context.Context, req *ingestor.TPulledDataRequest) (*ingestor.TDataProcessResponse, error) {
	target := c.numIDByHost[req.NumId]
	if target == nil {
		return nil, fmt.Errorf("unknown host for numId %d", req.NumId)
	}
	return target.WritePulled(ctx, req)
}

func (c *ingestorClientImpl) RefreshAssignments(ctx context.Context) error {
	c.numIDByHost = make(map[uint32]HostClient)

	for _, hostClient := range c.hostByName {
		req := ingestor.TGetAssignedShardsRequest{}
		resp, err := hostClient.GetAssignedShards(ctx, &req)
		if err != nil {
			return err
		}

		for numID := range resp.NumIds {
			c.numIDByHost[uint32(numID)] = hostClient
		}
	}

	log.Println("Success refresh assignments: ", len(c.numIDByHost))
	return nil
}
