package main

import (
	"context"
	"sync"
	"sync/atomic"
	"time"
)

type ClientsPool struct {
	sync.Mutex
	clients []IngestorClient
	idx     int32
}

func NewClientsPool() *ClientsPool {
	return &ClientsPool{}
}

func (c *ClientsPool) Init(target string, size int) error {
	c.Lock()
	defer c.Unlock()

	if len(c.clients) >= size {
		return nil
	}

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Minute)
	defer cancel()

	for i := len(c.clients); i < size; i++ {
		client, err := NewIngestorClient(ctx, target)
		if err != nil {
			return err
		}
		err = client.RefreshAssignments(ctx)
		if err != nil {
			return err
		}
		c.clients = append(c.clients, client)
	}
	return nil
}

func (c *ClientsPool) GetClient() IngestorClient {
	var nextIdx = int(atomic.AddInt32(&c.idx, 1)) % len(c.clients)
	if nextIdx < 0 {
		nextIdx = -nextIdx
	}
	return c.clients[nextIdx]
}
