package main

import (
	ingestor "a.yandex-team.ru/solomon/services/ingestor/api"
	"context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"
	"strconv"
	"time"
)

type HostClient interface {
	GetAssignedShards(ctx context.Context, req *ingestor.TGetAssignedShardsRequest) (*ingestor.TGetAssignedShardsResponse, error)
	WritePushed(ctx context.Context, req *ingestor.TPushedDataRequest) (*ingestor.TDataProcessResponse, error)
	WritePulled(ctx context.Context, req *ingestor.TPulledDataRequest) (*ingestor.TDataProcessResponse, error)
	Hostname() string
}

type hostClientImpl struct {
	hostname string
	client   ingestor.IngestorServiceClient
}

func NewHostClient(ctx context.Context, hostname, port string) (HostClient, error) {
	conn, err := grpc.DialContext(
		ctx,
		hostname+":"+port,
		grpc.WithInsecure(),
		grpc.WithUserAgent("short_term_storage_gun"))
	if err != nil {
		return nil, err
	}

	client := ingestor.NewIngestorServiceClient(conn)
	return &hostClientImpl{hostname: hostname, client: client}, nil
}

func appendRequiredMeta(ctx context.Context) context.Context {
	return metadata.AppendToOutgoingContext(ctx,
		"x-solomon-clientid", "short_term_storage_gun",
		"x-solomon-created-at", strconv.FormatInt(time.Now().UnixNano()/int64(time.Millisecond), 10))
}

func (h *hostClientImpl) Hostname() string {
	return h.hostname
}

func (h *hostClientImpl) GetAssignedShards(ctx context.Context, req *ingestor.TGetAssignedShardsRequest) (*ingestor.TGetAssignedShardsResponse, error) {
	ctx = appendRequiredMeta(ctx)
	return h.client.GetAssignedShards(ctx, req)
}

func (h *hostClientImpl) WritePushed(ctx context.Context, req *ingestor.TPushedDataRequest) (*ingestor.TDataProcessResponse, error) {
	ctx = appendRequiredMeta(ctx)
	return h.client.ProcessPushedData(ctx, req)
}

func (h *hostClientImpl) WritePulled(ctx context.Context, req *ingestor.TPulledDataRequest) (*ingestor.TDataProcessResponse, error) {
	ctx = appendRequiredMeta(ctx)
	return h.client.ProcessPulledData(ctx, req)
}
