package main

import (
	//"a.yandex-team.ru/library/go/core/log/zap"
	"context"
	"errors"
	"go.uber.org/zap"
	"time"

	"github.com/yandex/pandora/cli"
	"github.com/yandex/pandora/core"

	"github.com/golang/protobuf/proto"
	"github.com/yandex/pandora/core/aggregator/netsample"
	coreimport "github.com/yandex/pandora/core/import"
	"github.com/yandex/pandora/core/provider"
	"github.com/yandex/pandora/core/register"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/solomon/protos/common"
	ingestor "a.yandex-team.ru/solomon/services/ingestor/api"
)

var clientsPool = NewClientsPool()

type GunConfig struct {
	Target  string `validate:"required"`
	Clients int    `validate:"required"`
}

type Gun struct {
	conf GunConfig
	aggr core.Aggregator
	deps core.GunDeps
}

func NewGun(conf GunConfig) *Gun {
	return &Gun{conf: conf}
}

func (g *Gun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	err := clientsPool.Init(g.conf.Target, g.conf.Clients)
	if err != nil {
		return err
	}

	g.aggr = aggr
	g.deps = deps
	return nil
}

func (g *Gun) Shoot(genericAmmo core.Ammo) {
	ammo := genericAmmo.(*ProtoAmmo)
	sample := netsample.Acquire(ammo.Tag())

	var (
		err  error
		code = 500
	)

	defer func() {
		sample.SetProtoCode(code)
		if err != nil {
			g.deps.Log.Info("Request failed, status", zap.Int("code", code), zap.Error(err))
			sample.SetErr(err)
		}
		g.aggr.Report(sample)
	}()

	code, err = g.write(ammo)
}

func convertIngestorStatus(resp *ingestor.TDataProcessResponse) (int, error) {
	switch resp.Status[0] {
	case common.UrlStatusType_OK:
		return 200, nil

	case common.UrlStatusType_JSON_ERROR:
		fallthrough
	case common.UrlStatusType_SPACK_ERROR:
		fallthrough
	case common.UrlStatusType_PARSE_ERROR:
		fallthrough
	case common.UrlStatusType_DERIV_AND_TS:
		fallthrough
	case common.UrlStatusType_RESPONSE_TOO_LARGE:
		fallthrough
	case common.UrlStatusType_SENSOR_OVERFLOW:
		fallthrough
	case common.UrlStatusType_SHARD_IS_NOT_WRITABLE:
		return 400, errors.New(proto.MarshalTextString(resp))

	case common.UrlStatusType_QUOTA_ERROR:
		fallthrough
	case common.UrlStatusType_SKIP_TOO_LONG:
		return 413, nil

	case common.UrlStatusType_UNKNOWN_PROJECT:
		fallthrough
	case common.UrlStatusType_UNKNOWN_SHARD:
		return 404, nil

	case common.UrlStatusType_SHARD_NOT_INITIALIZED:
		fallthrough
	case common.UrlStatusType_IPC_QUEUE_OVERFLOW:
		return 503, nil

	case common.UrlStatusType_TIMEOUT:
		return 504, nil

	default:
		return 500, nil // errors.New(proto.MarshalTextString(resp))
	}
}

func convertGrpcStatus(err error) (int, error) {
	s := status.Convert(err)
	if s == nil {
		return 500, err
	}

	switch s.Code() {
	case codes.OK:
		return 200, nil
	case codes.Canceled:
		return 499, nil
	case codes.InvalidArgument:
		return 400, err
	case codes.DeadlineExceeded:
		return 504, err
	case codes.NotFound:
		return 404, err
	case codes.AlreadyExists:
		return 409, err
	case codes.PermissionDenied:
		return 403, err
	case codes.ResourceExhausted:
		return 429, err
	case codes.FailedPrecondition:
		return 400, err
	case codes.Aborted:
		return 409, err
	case codes.OutOfRange:
		return 400, err
	case codes.Unimplemented:
		return 501, err
	case codes.Unavailable:
		return 503, err
	case codes.Unauthenticated:
		return 401, err
	default:
		return 500, err
	}
}

func now() uint64 {
	return uint64(time.Now().UnixNano() / int64(time.Millisecond))
}

func (g *Gun) write(ammo *ProtoAmmo) (int, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	var err error
	var resp *ingestor.TDataProcessResponse
	var client = clientsPool.GetClient()
	if ammo.Pull {
		req := ammo.Message.(*ingestor.TPulledDataRequest)
		req.ResponseTimeMillis = append(req.ResponseTimeMillis, now())
		resp, err = client.WritePulled(ctx, req)
	} else {
		req := ammo.Message.(*ingestor.TPushedDataRequest)
		req.TimeMillis = now()
		resp, err = client.WritePushed(ctx, req)
	}

	if err != nil {
		return convertGrpcStatus(err)
	}

	return convertIngestorStatus(resp)
}

func main() {
	fs := coreimport.GetFs()
	coreimport.Import(fs)

	register.Provider("short_term_storage_ammo_provider", NewAmmoProvider, provider.DefaultJSONProviderConfig)
	register.Provider("short_term_storage_ammo_provider_preload", NewPreloadAmmoProvider, DefaultPreloadAmmoProviderConfig)
	register.Gun("short_term_storage_gun", NewGun, func() GunConfig {
		return GunConfig{}
	})

	cli.Run()
}
