#include <solomon/libs/cpp/slog/slog.h>
#include <solomon/libs/cpp/slog/canonical/lib/proto_decode.h>
#include <solomon/libs/cpp/slog/canonical/protos/metric.pb.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/generic/buffer.h>
#include <util/stream/buffer.h>
#include <util/stream/file.h>
#include <util/stream/format.h>
#include <util/stream/output.h>
#include <util/system/unaligned_mem.h>

#include <google/protobuf/text_format.h>

#include <array>

using namespace NMonitoring;
using namespace NSolomon::NSlog;
using yandex::monitoring::slog::TShardData;

namespace {

class TCliCommand {
public:
    TCliCommand()
        : Opts_(NLastGetopt::TOpts::Default())
    {
        Opts_.AddLongOption("data-out", "write data output into a specified file")
            .RequiredArgument("FILE");
        Opts_.AddLongOption("meta-out", "write meta output into a specified file")
            .RequiredArgument("FILE");
        Opts_.AddHelpOption();
        Opts_.SetFreeArgsNum(2);
        Opts_.SetFreeArgTitle(0, "<file>", "filename or '-' to read from stdin");
    }

    virtual ~TCliCommand() {
    }

    virtual TStringBuf Name() const noexcept = 0;
    virtual TStringBuf Help() const noexcept = 0;

    int Run(int argc, const char* argv[]) {
        Args_.Reset(new NLastGetopt::TOptsParseResult(&Opts_, argc, argv));
        return DoRun();
    }

protected:
    virtual int DoRun() = 0;

    TString ReadInputData() {
        auto freeArgs = Args_->GetFreeArgs();
        TString filename = freeArgs[1];
        if (filename == TStringBuf("-")) {
            return Cin.ReadAll();
        } else {
            return TUnbufferedFileInput(filename).ReadAll();
        }
    }

    IOutputStream* GetDataOutput() {
        DataOutput_.Reset(new TUnbufferedFileOutput(Args_->Get("data-out")));
        return DataOutput_.Get();
    }

    IOutputStream* GetMetaOutput() {
        MetaOutput_.Reset(new TUnbufferedFileOutput(Args_->Get("meta-out")));
        return MetaOutput_.Get();
    }

protected:
    NLastGetopt::TOpts Opts_;
    THolder<NLastGetopt::TOptsParseResult> Args_;
    THolder<IOutputStream> DataOutput_;
    THolder<IOutputStream> MetaOutput_;
};

class TFromProtoCmd: public TCliCommand {
public:
    TFromProtoCmd() {
        Opts_.AddLongOption(
            'c', "compress",
            "use specified compression algorithm when writing in slog. "
            "Possible values are: { IDENTITY, ZLIB, ZSTD, LZ4 }")
        .Optional()
        .RequiredArgument("ALG")
        .DefaultValue("ZSTD");
    }

private:
    TStringBuf Name() const noexcept override {
        return TStringBuf("from-proto");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("converts metrics from proto into slog");
    }

    int DoRun() override {
        TString dataStr = ReadInputData();
        TShardData shardData;
        bool isParsedCorrectly = google::protobuf::TextFormat::ParseFromString(dataStr, &shardData);
        Y_ENSURE(isParsedCorrectly, "failed to parse a proto msg");

        auto encoder = CreateSlogEncoder(
                shardData.GetNumId(),
                ETimePrecision::MILLIS,
                FromString<ECompression>(Args_->Get("compress")),
                GetDataOutput(),
                GetMetaOutput());

        DecodeProto(shardData, encoder.Get());
        return 0;
    }
};

std::array<TCliCommand*, 1> Commands = {{
    Singleton<TFromProtoCmd>(),
// TODO: more commands:
//    Singleton<TToProtoCmd>(),
//    Singleton<TMetaHeaderCmd>(),
//    Singleton<TDataHeaderCmd>(),
}};

void PrintUsage(const char* appName) {
    size_t maxLen = 0;
    for (const TCliCommand* cmd: Commands) {
        maxLen = Max(maxLen, cmd->Name().size());
    }

    Cerr << appName << " [-h] <command> [OPTIONS]\n\n"
         << "Known commands are:\n";

    for (const TCliCommand* cmd: Commands) {
        Cerr << "  * " << RightPad(cmd->Name(), maxLen, ' ')
             << "    " << cmd->Help() << '\n';
    }

    Cerr << Flush;
}

} // namespace

int main(int argc, const char* argv[]) {
    try {
        if (argc <= 1) {
            PrintUsage(argv[0]);
            return 1;
        }

        TStringBuf cmdName = argv[1];
        if (cmdName == TStringBuf("-h") || cmdName == TStringBuf("--help")) {
            PrintUsage(argv[0]);
            return 0;
        }

        for (TCliCommand* cmd: Commands) {
            if (cmdName == cmd->Name()) {
                return cmd->Run(argc, argv);
            }
        }

        Cerr << "Unknown command: '" << cmdName << '\'' << Endl;
        PrintUsage(argv[0]);
        return 1;
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}
