package main

import (
	"a.yandex-team.ru/solomon/tools/solomon-sync/juggler"
	"fmt"
	"log"
	"strings"
)

func (c *SolomonClient) ListAlertIds(instance *Instance) ([]string, error) {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/alerts?pageSize=30"
	resp, err := c.Get(instance, endpoint)
	if err != nil {
		return nil, err
	}
	result := make([]string, 0)
	for {
		ids, nextToken, err := ParseListIdsWithToken(resp)
		if err != nil {
			return nil, err
		}
		result = append(result, ids...)
		if nextToken == "" {
			break
		}

		resp, err = c.Get(instance, endpoint+"&pageToken="+nextToken)
		if err != nil {
			return nil, err
		}
	}

	return result, nil
}

func (c *SolomonClient) GetAlert(instance *Instance, id string) (map[string]interface{}, error) {
	body, err := c.GetOptional(instance, "/api/v2/projects/"+instance.ProjectID+"/alerts/"+id)
	if err != nil {
		return nil, err
	}
	if body == nil {
		return nil, nil
	}
	return deserialize(body)
}

func (c *SolomonClient) CreateAlert(instance *Instance, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/alerts"
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Post(instance, endpoint, body)
}

func (c *SolomonClient) UpdateAlert(instance *Instance, id string, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/alerts/" + id
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Put(instance, endpoint, body)
}

func (c *SolomonClient) DeleteAlert(instance *Instance, id string) error {
	return c.Delete(instance, "/api/v2/projects/"+instance.ProjectID+"/alerts/"+id)
}

func (c *SolomonClient) SyncAlerts(p Plan) {
	source, err := c.ListAlertIds(ProdMainSolomon)
	if err != nil {
		log.Println("Failed to ListAlertIds: " + err.Error())
		return
	}
	for _, id := range source {
		entity, err := c.GetAlert(ProdMainSolomon, id)
		if err != nil {
			log.Println("Failed to GetAlert " + id + ": " + err.Error())
			continue
		}
		if entity == nil {
			log.Println(ProdMainSolomon.Name + " Alert " + id + " already removed from source")
			continue
		}

		p.SyncInstances(func(instance *Instance) {
			if isIgnored(instance, entity) {
				log.Println(instance.Name + " Alert " + id + " SKIP marked as ignore on source")
			} else if instance.EnableAlert {
				c.syncAlert(instance, id, entity)
			}
		})
	}
	p.SyncInstances(func(instance *Instance) {
		if instance.EnableAlert {
			c.deleteNotExistAlerts(instance, source)
		}
	})
}

func (c *SolomonClient) updateJugglerCheck(instance *Instance, patchedAlert map[string]interface{}) {
	alertID := patchedAlert["id"].(string)
	check := juggler.Check{
		Project: "solomon",
		Host:    instance.JugglerHost,
		Service: alertID,
		Tags:    instance.JugglerTags,
		Children: []juggler.Child{
			{
				Type:     "EVENTS",
				Host:     fmt.Sprintf("service=%s", instance.JugglerRawEventsServicePrefix+alertID),
				Service:  "all",
				Instance: "all",
			},
		},
		Description:      "generated by solomon-sync",
		TTL:              600,
		Aggregator:       "logic_or",
		AggregatorKwargs: map[string]string{"nodata_mode": "force_ok"},
		RefreshTime:      90,
		Meta: juggler.Meta{
			Urls: []juggler.URL{
				{
					Title: "Alert in solomon",
					URL:   instance.Host + "/admin/projects/" + instance.ProjectID + "/alerts/" + alertID,
				},
			},
		},
	}
	err := c.Juggler.UpsertCheck(&check)
	if err != nil {
		log.Println(instance.Name + " Failed to update juggler check for alert " + alertID + ": " + err.Error())
	}
}

func (c *SolomonClient) removeJugglerCheck(instance *Instance, alertID string) {
	check := juggler.Check{
		Project: "solomon",
		Host:    instance.JugglerHost,
		Service: alertID,
	}
	err := c.Juggler.RemoveCheck(&check)
	if err != nil {
		log.Println(instance.Name + " Failed to update juggler check for alert " + alertID + ": " + err.Error())
	}
}

func (c *SolomonClient) syncAlert(instance *Instance, id string, source map[string]interface{}) {
	patched, createJugglerChecks := patchAlert(instance, source)
	target, err := c.GetAlert(instance, id)
	if err != nil {
		log.Println(instance.Name + " Alert " + id + " FAILED: " + err.Error())
		return
	}
	if target == nil {
		if err = c.CreateAlert(instance, patched); err == nil {
			log.Println(instance.Name + " Alert " + id + " CREATED")
			if createJugglerChecks {
				c.updateJugglerCheck(instance, patched)
			}
		} else {
			log.Println(instance.Name + " Alert " + id + " FAILED: " + err.Error())
		}
	} else if isIgnored(instance, target) {
		log.Println(instance.Name + " Alert " + id + " SKIP because marked as ignore")
	} else {
		RemoveCommonAttributes(target)
		CopyVersion(target, patched)

		if isEqualLogDiff(instance, "alert", patched, target) {
			log.Println(instance.Name + " Alert " + id + " SKIP already sync")
		} else {
			if err = c.UpdateAlert(instance, id, patched); err == nil {
				log.Println(instance.Name + " Alert " + id + " UPDATED")
				if createJugglerChecks {
					c.updateJugglerCheck(instance, patched)
				}
			} else {
				log.Println(instance.Name + " Alert " + id + " FAILED: " + err.Error())
			}
		}
	}
}

func RemoveNotificationChannels(target map[string]interface{}) {
	delete(target, "notificationChannels")
	delete(target, "channels")
}

func AddJugglerChannel(target map[string]interface{}) {
	target["notificationChannels"] = []string{"juggler"}
}

func (c *SolomonClient) deleteNotExistAlerts(instance *Instance, ids []string) {
	unique := make(map[string]bool)
	for _, id := range ids {
		unique[id] = true
	}

	targetIds, err := c.ListAlertIds(instance)
	if err != nil {
		log.Println(instance.Name + " Failed to ListAlertIds: " + err.Error())
		return
	}
	for _, id := range targetIds {
		exist := unique[id]
		if !exist {
			entity, err := c.GetAlert(instance, id)
			if err != nil {
				log.Println(instance.Name + " Alert " + id + " FAILED because: " + err.Error())
				continue
			}
			if entity == nil {
				continue
			} else if isIgnored(instance, entity) {
				log.Println(instance.Name + " Alert " + id + " SKIP because mark as ignore")
				continue
			}

			if err = c.DeleteAlert(instance, id); err == nil {
				log.Println(instance.Name + " Alert " + id + " DELETED")
				c.removeJugglerCheck(instance, id)
			} else {
				log.Println(instance.Name + " Alert " + id + " FAILED because: " + err.Error())
			}
		}
	}
}

func patchAlert(instance *Instance, source map[string]interface{}) (map[string]interface{}, bool) {
	patched := applyMap(source, "", func(path string, sourceStr string) string {
		if sourceStr == "solomon" {
			return instance.ProjectID
		}
		replacedStr := sourceStr
		replacedStr = ApplyAllRules(instance, replacedStr)
		replacedStr = strings.ReplaceAll(replacedStr, "\"solomon\"", "\""+instance.ProjectID+"\"")
		replacedStr = strings.ReplaceAll(replacedStr, "'solomon'", "'"+instance.ProjectID+"'")
		if path == ".annotations.host" {
			replacedStr = strings.ReplaceAll(replacedStr, "solomon", instance.ProjectID)
		}
		replacedStr = PathReference(instance, replacedStr)
		return replacedStr
	})

	RemoveCommonAttributes(patched)
	Disclaimer("alert", patched)

	name := patched["name"].(string)
	patched["name"] = instance.AlertPrefix + name

	createJugglerChecks := false
	switch instance.EnableChannel {
	case Disable:
		RemoveNotificationChannels(patched)
	case ToJuggler:
		hasSolomonOps := false
		if channels, ok := patched["notificationChannels"]; ok {
			for _, channel := range channels.([]interface{}) {
				if channel.(string) == "solomon-ops-telegram" {
					hasSolomonOps = true
				}
			}
		}
		RemoveNotificationChannels(patched)
		if hasSolomonOps {
			AddJugglerChannel(patched)
			createJugglerChecks = true
		}
	}
	delete(patched, "labels")
	delete(patched, "serviceProviderAnnotations")

	return patched, createJugglerChecks
}
