package main

import (
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/solomon/tools/solomon-sync/juggler"
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"strconv"
	"strings"
	"time"
)

type SolomonClient struct {
	Token    string
	IAMToken string
	HTTP     http.Client
	DryRun   bool
	Registry *solomon.Registry
	Juggler  *juggler.Client
}

func splitPath(url string) (string, string) {
	tokens := strings.Split(url, "/")
	projectID := tokens[4]
	endpoint := tokens[5]
	if len(tokens) > 5 {
		endpoint = endpoint + "/:id"
	}
	return projectID, "/api/v2/projects/:projectId/" + endpoint
}

var expBuckets = metrics.MakeExponentialBuckets(16, 2, 12)

func (c *SolomonClient) Do(instance *Instance, req *http.Request) (*http.Response, error) {
	req.Header.Add("Authorization", instance.GetToken(c))
	req.Header.Add("Content-Type", "application/json")
	attempt := 0
	projectID, endpoint := splitPath(req.URL.Path)

	var regs = make([]metrics.Registry, 0, 16)

	for _, method := range [2]string{req.Method, "total"} {
		for _, instance := range [2]string{req.URL.Host, "total"} {
			for _, prj := range [2]string{projectID, "total"} {
				for _, endp := range [2]string{endpoint, "total"} {
					regs = append(regs, c.Registry.WithTags(map[string]string{
						"method":    method,
						"instance":  instance,
						"projectId": prj,
						"endpoint":  endp,
					}))
				}
			}
		}
	}

	for {
		start := time.Now()
		resp, err := c.HTTP.Do(req)
		stop := time.Now()
		elapsedMillis := stop.Sub(start).Milliseconds()

		for _, reg := range regs {
			reg.Histogram("http.client.call.elapsedTimeMs", expBuckets).RecordValue(float64(elapsedMillis))
		}

		if err != nil {
			for _, reg := range regs {
				reg.Gauge("http.client.call.completedError").Add(1)
			}
			attempt++
			if attempt < 3 {
				time.Sleep(5 * time.Second)
				continue
			}

			return nil, err
		}

		for _, reg := range regs {
			reg.Gauge("http.client.call.completedOk").Add(1)
			codereg := reg.WithTags(map[string]string{"code": strconv.Itoa(resp.StatusCode)})
			codereg.Gauge("http.client.call.status").Add(1)
		}

		return resp, nil
	}
}

func (c *SolomonClient) Get(instance *Instance, path string) ([]byte, error) {
	url := instance.Host + path
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		return nil, fmt.Errorf("bad GET request: "+url+": %w", err)
	}
	resp, err := c.Do(instance, req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)

	if err != nil {
		return nil, fmt.Errorf("Error reading GET: "+url+": %w", err)
	}
	if resp.StatusCode != 200 {
		log.Println(string(body))
		return nil, fmt.Errorf("GET %s returned %s", req.URL.String(), resp.Status)
	}
	return body, nil
}

func (c *SolomonClient) Post(instance *Instance, path string, body []byte) error {
	url := instance.Host + path
	if c.DryRun {
		log.Print("Simulating POST to " + url)
		return nil
	}
	req, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
	if err != nil {
		return fmt.Errorf("bad POST request: "+url+": %w", err)
	}
	resp, err := c.Do(instance, req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		respBody, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			log.Println(string(respBody))
		}
		return fmt.Errorf("POST %s returned %s", req.URL.String(), resp.Status)
	}
	return nil
}

func (c *SolomonClient) Put(instance *Instance, path string, body []byte) error {
	url := instance.Host + path
	if c.DryRun {
		log.Print("Simulating PUT to " + url)
		return nil
	}
	req, err := http.NewRequest("PUT", url, bytes.NewBuffer(body))
	if err != nil {
		return fmt.Errorf("bad PUT request: "+url+": %w", err)
	}
	resp, err := c.Do(instance, req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		respBody, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			log.Println(string(respBody))
		}
		return fmt.Errorf("PUT %s returned %s", req.URL.String(), resp.Status)
	}
	return nil
}

func (c *SolomonClient) Delete(instance *Instance, path string) error {
	url := instance.Host + path
	if c.DryRun {
		log.Print("Simulating DELETE to " + url)
		return nil
	}
	req, err := http.NewRequest("DELETE", url, nil)
	if err != nil {
		return fmt.Errorf("bad DELETE request: "+url+": %w", err)
	}
	resp, err := c.Do(instance, req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 && resp.StatusCode != 204 {
		respBody, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			log.Println(string(respBody))
		}
		return fmt.Errorf("DELETE %s returned %s", req.URL.String(), resp.Status)
	}
	return nil
}

func (c *SolomonClient) GetOptional(instance *Instance, path string) ([]byte, error) {
	url := instance.Host + path
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		return nil, fmt.Errorf("bad GET: "+url+": %w", err)
	}
	resp, err := c.Do(instance, req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode == 404 {
		return nil, nil
	} else if resp.StatusCode != 200 {
		return nil, fmt.Errorf("GET %s returned %s", req.URL.String(), resp.Status)
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("Error reading GET: "+url+": %w", err)
	}
	return body, nil
}

func printJSON(body []byte) error {
	var prettyJSON bytes.Buffer
	err := json.Indent(&prettyJSON, body, "", "  ")
	if err != nil {
		return fmt.Errorf(string(body)+": %w", err)
	}

	log.Println(prettyJSON.String())

	return nil
}
