package main

import (
	"log"
)

func (c *SolomonClient) ListDashboardIds(instance *Instance) ([]string, error) {
	resp, err := c.Get(instance, "/api/v2/projects/"+instance.ProjectID+"/dashboards?pageSize=all")
	if err != nil {
		return nil, err
	}
	return ParseListIds(resp)
}

func (c *SolomonClient) GetDashboard(instance *Instance, id string) (map[string]interface{}, error) {
	body, err := c.GetOptional(instance, "/api/v2/projects/"+instance.ProjectID+"/dashboards/"+id)
	if err != nil {
		return nil, err
	}
	if body == nil {
		return nil, nil
	}
	return deserialize(body)
}

func (c *SolomonClient) CreateDashboard(instance *Instance, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/dashboards"
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Post(instance, endpoint, body)
}

func (c *SolomonClient) UpdateDashboard(instance *Instance, id string, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/dashboards/" + id
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Put(instance, endpoint, body)
}

func (c *SolomonClient) DeleteDashboard(instance *Instance, id string) error {
	return c.Delete(instance, "/api/v2/projects/"+instance.ProjectID+"/dashboards/"+id)
}

func (c *SolomonClient) SyncDashboards(p Plan) {
	ids, err := c.ListDashboardIds(ProdMainSolomon)
	if err != nil {
		log.Println("Failed to ListDashboardIds: " + err.Error())
		return
	}
	for _, id := range ids {
		entity, err := c.GetDashboard(ProdMainSolomon, id)
		if err != nil {
			log.Println("Failed to GetDashboard " + id + ": " + err.Error())
			continue
		}
		if entity == nil {
			log.Println("Dashboard" + id + " already removed from source")
			continue
		}

		p.SyncInstances(func(instance *Instance) {
			c.syncDashboard(instance, id, entity)
		})
	}

	p.SyncInstances(func(instance *Instance) {
		c.deleteNotExistDashboards(instance, ids)
	})
}

func (c *SolomonClient) syncDashboard(instance *Instance, id string, source map[string]interface{}) {
	patched := patchDashboard(instance, id, source)
	syncID := patched["id"].(string)
	target, err := c.GetDashboard(instance, syncID)
	if err != nil {
		log.Println(instance.Name + " Dashboard " + syncID + " FAILED: " + err.Error())
		return
	}
	if target == nil {
		if err = c.CreateDashboard(instance, patched); err == nil {
			log.Println(instance.Name + " Dashboard " + syncID + " CREATED")
		} else {
			log.Println(instance.Name + " Dashboard " + syncID + " FAILED: " + err.Error())
		}
	} else {
		RemoveCommonAttributes(target)
		DropGeneratedID(target)
		CopyVersion(target, patched)
		if isEqualLogDiff(instance, "dashboard", patched, target) {
			log.Println(instance.Name + " Dashboard " + syncID + " SKIP already sync")
		} else {
			if err = c.UpdateDashboard(instance, syncID, patched); err == nil {
				log.Println(instance.Name + " Dashboard " + syncID + " UPDATED")
			} else {
				log.Println(instance.Name + " Dashboard " + syncID + " FAILED: " + err.Error())
			}
		}
	}
}

func (c *SolomonClient) deleteNotExistDashboards(instance *Instance, ids []string) {
	unique := make(map[string]bool)
	for _, id := range ids {
		unique[instance.Prefix+id] = true
	}

	targetIds, err := c.ListDashboardIds(instance)
	if err != nil {
		log.Println("Failed to ListDashboardIds: " + err.Error())
		return
	}
	for _, id := range targetIds {
		exist := unique[id]
		if !exist {
			if err = c.DeleteDashboard(instance, id); err == nil {
				log.Println(instance.Name + " Dashboard " + id + " DELETED")
			} else {
				log.Println(instance.Name + " Dashboard " + id + " FAILED: " + err.Error())
			}
		}
	}
}

func patchDashboard(instance *Instance, id string, source map[string]interface{}) map[string]interface{} {
	patched := applyMap(source, "", func(path string, s string) string {
		if s == id {
			return instance.Prefix + id
		}
		if s == "solomon" {
			return instance.ProjectID
		}
		s = ApplyAllRules(instance, s)
		return PathReference(instance, s)
	})
	RemoveCommonAttributes(patched)
	DropGeneratedID(patched)
	return patched
}
