package main

import (
	"log"
	"strings"
)

func (c *SolomonClient) ListGraphIds(instance *Instance) ([]string, error) {
	resp, err := c.Get(instance, "/api/v2/projects/"+instance.ProjectID+"/graphs?pageSize=all")
	if err != nil {
		return nil, err
	}
	return ParseListIds(resp)
}

func (c *SolomonClient) GetGraph(instance *Instance, id string) (map[string]interface{}, error) {
	body, err := c.GetOptional(instance, "/api/v2/projects/"+instance.ProjectID+"/graphs/"+id)
	if err != nil {
		return nil, err
	}
	if body == nil {
		return nil, nil
	}
	return deserialize(body)
}

func (c *SolomonClient) CreateGraph(instance *Instance, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/graphs"
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Post(instance, endpoint, body)
}

func (c *SolomonClient) UpdateGraph(instance *Instance, graphID string, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/graphs/" + graphID
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Put(instance, endpoint, body)
}

func (c *SolomonClient) DeleteGraph(instance *Instance, graphID string) error {
	return c.Delete(instance, "/api/v2/projects/"+instance.ProjectID+"/graphs/"+graphID)
}

func (c *SolomonClient) SyncGraphs(p Plan) {
	source, err := c.ListGraphIds(ProdMainSolomon)
	if err != nil {
		log.Println("Failed to ListGraphIds: " + err.Error())
		return
	}
	for _, graphID := range source {
		graph, err := c.GetGraph(ProdMainSolomon, graphID)
		if err != nil {
			log.Println("Failed to GetGraph " + graphID + ": " + err.Error())
			continue
		}
		if graph == nil {
			log.Println("Graph" + graphID + " already removed from source")
			continue
		}

		p.SyncInstances(func(instance *Instance) {
			c.syncGraph(instance, graphID, graph)
		})
	}
	p.SyncInstances(func(instance *Instance) {
		c.deleteNotExistGraphs(instance, source)
	})
}

func (c *SolomonClient) syncGraph(instance *Instance, graphID string, source map[string]interface{}) {
	patched := patchGraph(instance, graphID, source)
	target, err := c.GetGraph(instance, instance.Prefix+graphID)
	if err != nil {
		log.Println(instance.Name + " Graph " + graphID + " FAILED: " + err.Error())
		return
	}
	if target == nil {
		if err = c.CreateGraph(instance, patched); err == nil {
			log.Println(instance.Name + " Graph " + graphID + " CREATED")
		} else {
			log.Println(instance.Name + " Graph " + graphID + " FAILED: " + err.Error())
		}
	} else {
		RemoveCommonAttributes(target)
		DropGeneratedID(target)
		CopyVersion(target, patched)
		if isEqualLogDiff(instance, "graph", patched, target) {
			log.Println(instance.Name + " Graph " + graphID + " SKIP already sync")
		} else {
			if err = c.UpdateGraph(instance, instance.Prefix+graphID, patched); err == nil {
				log.Println(instance.Name + " Graph " + graphID + " UPDATED")
			} else {
				log.Println(instance.Name + " Graph " + graphID + " FAILED: " + err.Error())
			}
		}
	}
}

func (c *SolomonClient) deleteNotExistGraphs(instance *Instance, graphIds []string) {
	unique := make(map[string]bool)
	for _, graphID := range graphIds {
		unique[instance.Prefix+graphID] = true
	}

	targetGraphIds, err := c.ListGraphIds(instance)
	if err != nil {
		log.Println("Failed to ListGraphIds: " + err.Error())
		return
	}
	for _, graphID := range targetGraphIds {
		exist := unique[graphID]
		if !exist {
			if err = c.DeleteGraph(instance, graphID); err == nil {
				log.Println(instance.Name + " Graph " + graphID + " DELETED")
			} else {
				log.Println(instance.Name + " Graph " + graphID + " FAILED: " + err.Error())
			}
		}
	}
}

func patchGraph(instance *Instance, graphID string, source map[string]interface{}) map[string]interface{} {
	patched := applyMap(source, "", func(path string, s string) string {
		if s == "solomon" {
			return instance.ProjectID
		}
		if s == graphID {
			return instance.Prefix + graphID
		}
		patchedStr := s
		patchedStr = strings.ReplaceAll(patchedStr, "prestable", instance.Pre)
		return patchedStr
	})
	RemoveCommonAttributes(patched)
	DropGeneratedID(patched)
	Disclaimer("graph", patched)
	return patched
}
