package main

import "regexp"

type ChannelMapping int

const (
	Disable ChannelMapping = iota
	ToJuggler
	OneToOne
)

type ReplaceRule struct {
	From *regexp.Regexp
	To   string
}

// Instance describes how different solomon installations are monitored.
// An instance is monitored in a host installation
type Instance struct {
	// Name is a unique name of an instance
	Name string
	// Host holds API root url of the hosting installation
	Host string
	// IAMBased is true if IAM token is used instead of OAuth
	IAMBased bool
	// ProjectID which is used in the hosting installation for this instance
	ProjectID string
	// Pre is the term used for the prestable installation
	Pre string
	// Prefix is added to graphs and dashboards. In the old days graph and dashboard ids had to be globally unique
	Prefix string
	// EnableChannel controls how to generate notification channels in alerts and to sync them
	EnableChannel ChannelMapping
	// AlertPrefix is used as a part of alert name to distinguish for alerts from different instances if they are
	// delivered to the same channel
	AlertPrefix string
	// EnableAlert controls whether to sync alerts or not
	EnableAlert bool
	// Admin url is used to patch staff-only urls in menu and alerts
	Admin string
	// AdminPre url is used to patch staff-only urls in menu and alerts for prestable links
	AdminPre string
	// SkipPre is true if instance has no prestable and does not need prestable entities (with pre- prefix)
	SkipPre bool
	// ReplicaPatch rules to patch replicas
	ReplicaPatch []ReplaceRule
	// JugglerHost is a host used in aggregate checks for this instance
	JugglerHost string
	// JugglerRawEventsServicePrefix is a prefix used in service field for raw events
	JugglerRawEventsServicePrefix string
	// JugglerTags is a list of tags for aggregate checks (meaningful only when EnableChannel is ToJuggler)
	JugglerTags []string
}

var ProdMainSolomon = &Instance{
	Name:          "PROD_MAIN_SOLOMON",
	Host:          "https://solomon.yandex-team.ru",
	ProjectID:     "solomon",
	Pre:           "prestable",
	Prefix:        "",
	EnableChannel: OneToOne,
	EnableAlert:   true,
	Admin:         "https://solomon.yandex-team.ru",
	AdminPre:      "https://solomon-pre.yandex-team.ru",
}

var ProdMainSolomonCloud = &Instance{
	Name:          "PROD_MAIN_SOLOMON_CLOUD",
	Host:          "https://solomon.yandex-team.ru",
	ProjectID:     "solomon_cloud",
	Pre:           "preprod",
	Prefix:        "sync-",
	EnableChannel: OneToOne,
	AlertPrefix:   "[Cloud] ",
	EnableAlert:   true,
	Admin:         "https://solomon.cloud.yandex-team.ru",
	AdminPre:      "https://solomon.cloud-preprod.yandex-team.ru",
}

var PreMainSolomon = &Instance{
	Name:        "PRE_MAIN_SOLOMON",
	Host:        "https://solomon-pre.yandex-team.ru",
	ProjectID:   "solomon",
	Pre:         "prestable",
	Prefix:      "",
	AlertPrefix: "[Pre] ",
	EnableAlert: true,
	Admin:       "https://solomon.yandex-team.ru",
	AdminPre:    "https://solomon-pre.yandex-team.ru",
}

var PreMainSolomonCloud = &Instance{
	Name:        "PRE_MAIN_SOLOMON_CLOUD",
	Host:        "https://solomon-pre.yandex-team.ru",
	ProjectID:   "solomon_cloud",
	Pre:         "preprod",
	Prefix:      "sync-",
	AlertPrefix: "[Pre] ",
	EnableAlert: true,
	Admin:       "https://solomon.cloud.yandex-team.ru",
	AdminPre:    "https://solomon.cloud-preprod.yandex-team.ru",
}

var ProdCloudSolomon = &Instance{
	Name:        "PROD_CLOUD_SOLOMON",
	Host:        "https://solomon.cloud.yandex-team.ru",
	ProjectID:   "solomon",
	Pre:         "preprod",
	Prefix:      "",
	AlertPrefix: "[Cloud] ",
	EnableAlert: false,
	Admin:       "https://solomon.cloud.yandex-team.ru",
	AdminPre:    "https://solomon.cloud-preprod.yandex-team.ru",
}

var PreCloudSolomon = &Instance{
	Name:        "PRE_CLOUD_SOLOMON",
	Host:        "https://solomon.cloud-preprod.yandex-team.ru",
	ProjectID:   "solomon",
	Pre:         "preprod",
	Prefix:      "",
	AlertPrefix: "[Preprod] ",
	EnableAlert: true,
	Admin:       "https://solomon.cloud.yandex-team.ru",
	AdminPre:    "https://solomon.cloud-preprod.yandex-team.ru",
}

var ProdGpn = &Instance{
	Name:          "PROD_GPN",
	Host:          "https://solomon.ycp.gpn.yandexcloud.net",
	IAMBased:      true,
	ProjectID:     "yc.monitoring.cloud",
	Pre:           "production", // gpn has no prestable but prestable is present in dashboards and alerts
	Prefix:        "",
	AlertPrefix:   "[GPN] ",
	EnableChannel: ToJuggler,
	EnableAlert:   true,
	Admin:         "https://solomon.ycp.gpn.yandexcloud.net",
	AdminPre:      "https://solomon.ycp.gpn.yandexcloud.net",
	SkipPre:       true,
	ReplicaPatch: []ReplaceRule{
		{regexp.MustCompile(`_sas`), "_a"},
		{regexp.MustCompile(`_vla`), "_b"},
		{regexp.MustCompile(`-sas-`), "-a-"},
		{regexp.MustCompile(`-vla-`), "-b-"},
		{regexp.MustCompile(`avgSas`), "avgRepA"},
		{regexp.MustCompile(`avgVla`), "avgRepB"},
		{regexp.MustCompile(`stdSas`), "stdRepA"},
		{regexp.MustCompile(`stdVla`), "stdRepB"},
		{regexp.MustCompile(`integrateSas`), "integrateRepA"},
		{regexp.MustCompile(`integrateVla`), "integrateRepB"},
		{regexp.MustCompile(`Sas`), "replica-a"},
		{regexp.MustCompile(`Vla`), "replica-b"},
		{regexp.MustCompile(`\?\?\?`), "replica-?"},
		{regexp.MustCompile(`\bSAS\b`), "REPLICA_A"},
		{regexp.MustCompile(`\bVLA\b`), "REPLICA_B"},
	},
	JugglerTags:                   []string{"solomon-ops-gpn"},
	JugglerHost:                   "solomon-gpn",
	JugglerRawEventsServicePrefix: "gpn-",
}

var ProdIsrael = &Instance{
	Name:          "PROD_ISRAEL",
	Host:          "https://solomon.yandexcloud.co.il",
	IAMBased:      true,
	ProjectID:     "yc.monitoring.cloud",
	Pre:           "prestable", // no need to replace
	Prefix:        "",
	AlertPrefix:   "[IL] ",
	EnableChannel: ToJuggler,
	EnableAlert:   true,
	Admin:         "https://solomon.yandexcloud.co.il",
	AdminPre:      "https://solomon.yandexcloud.co.il",
	SkipPre:       true,
	ReplicaPatch: []ReplaceRule{
		{regexp.MustCompile(`_sas`), "_a"},
		{regexp.MustCompile(`_vla`), "_b"},
		{regexp.MustCompile(`-sas-`), "-a-"},
		{regexp.MustCompile(`-vla-`), "-b-"},
		{regexp.MustCompile(`avgSas`), "avgRepA"},
		{regexp.MustCompile(`avgVla`), "avgRepB"},
		{regexp.MustCompile(`stdSas`), "stdRepA"},
		{regexp.MustCompile(`stdVla`), "stdRepB"},
		{regexp.MustCompile(`integrateSas`), "integrateRepA"},
		{regexp.MustCompile(`integrateVla`), "integrateRepB"},
		{regexp.MustCompile(`Sas`), "replica-a"},
		{regexp.MustCompile(`Vla`), "replica-b"},
		{regexp.MustCompile(`\?\?\?`), "replica-?"},
		{regexp.MustCompile(`\bSAS\b`), "REPLICA_A"},
		{regexp.MustCompile(`\bVLA\b`), "REPLICA_B"},
		{regexp.MustCompile(`solomon-core-`), "core-"},
	},
	JugglerTags:                   []string{"solomon-ops-il"},
	JugglerHost:                   "solomon-il",
	JugglerRawEventsServicePrefix: "il-",
}

var instances = make(map[string]*Instance)

func register(instance *Instance) {
	instances[instance.Name] = instance
}

func init() {
	register(ProdMainSolomon)
	register(ProdMainSolomonCloud)
	register(PreMainSolomon)
	register(PreMainSolomonCloud)
	register(ProdCloudSolomon)
	register(PreCloudSolomon)
	register(ProdIsrael)
}

func (p Plan) SyncInstances(fn func(instance *Instance)) {
	for _, name := range p {
		fn(instances[name])
	}
}

func (i *Instance) GetToken(client *SolomonClient) string {
	if i.IAMBased {
		return "Bearer " + client.IAMToken
	}
	return "OAuth " + client.Token
}
