package juggler

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"time"
)

type Client struct {
	Endpoint string
	Token    string
	HTTP     http.Client
}

func MakeClient(endpoint string, token string) *Client {
	if endpoint == "" {
		endpoint = "https://juggler-api.search.yandex.net/api/"
	}
	return &Client{
		Endpoint: endpoint,
		Token:    token,
		HTTP:     http.Client{Timeout: 30 * time.Second},
	}
}

type Child struct {
	Type     string `json:"type"`
	Host     string `json:"host"`
	Service  string `json:"service"`
	Instance string `json:"instance"`
}

type URL struct {
	Title string `json:"title"`
	URL   string `json:"url"`
	Type  string `json:"type"`
}

type Meta struct {
	Urls []URL `json:"urls"`
}

type Check struct {
	Project          string            `json:"project"`
	Host             string            `json:"host"`
	Service          string            `json:"service"`
	Tags             []string          `json:"tags"`
	Children         []Child           `json:"children"`
	Description      string            `json:"description"`
	TTL              int               `json:"ttl"`
	Aggregator       string            `json:"aggregator"`
	AggregatorKwargs map[string]string `json:"aggregator_kwargs"`
	RefreshTime      int               `json:"refresh_time"`
	Meta             Meta              `json:"meta"`
}

func (client *Client) UpsertCheck(check *Check) error {
	body, err := json.Marshal(check)
	if err != nil {
		return fmt.Errorf("failed to serialize check: %w", err)
	}
	url := client.Endpoint + "checks/add_or_update?do=1"
	req, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
	req.Header.Add("Content-Type", "application/json")
	req.Header.Add("Authorization", "OAuth "+client.Token)
	if err != nil {
		return fmt.Errorf("bad POST request: %s: %w", url, err)
	}
	resp, err := client.HTTP.Do(req)
	if err != nil {
		return fmt.Errorf("failed to upsert juggler check: %w", err)
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		respBody, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			log.Println(string(respBody))
		}
		return fmt.Errorf("upserting juggler check returned %s", resp.Status)
	}
	return nil
}

func (client *Client) RemoveCheck(check *Check) error {
	url := fmt.Sprintf("%schecks/remove_check?project=%s&host_name=%s&service_name=%s&do=1",
		client.Endpoint, check.Project, check.Host, check.Service)
	req, err := http.NewRequest("GET", url, bytes.NewBuffer([]byte{}))
	req.Header.Add("Authorization", "OAuth "+client.Token)
	if err != nil {
		return fmt.Errorf("bad GET request: %s: %w", url, err)
	}
	resp, err := client.HTTP.Do(req)
	if err != nil {
		return fmt.Errorf("failed to upsert juggler check: %w", err)
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 && resp.StatusCode != 204 {
		respBody, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			log.Println(string(respBody))
		}
		return fmt.Errorf("upserting juggler check returned %s", resp.Status)
	}
	return nil
}
