package main

import (
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/solomon/tools/solomon-sync/juggler"
	"context"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"strings"
	"time"
)

func ReadTokenFromFile(fileName string) (string, error) {
	b, err := ioutil.ReadFile(fileName)
	if err != nil {
		return "", err
	}
	return strings.TrimSuffix(string(b), "\n"), nil
}

func chooseToken(token, tokenFile *string, envVar string) (string, error) {
	tokenEnv := os.Getenv(envVar)
	if *token != "" {
		return *token, nil
	} else if *tokenFile != "" {
		return ReadTokenFromFile(*tokenFile)
	} else if tokenEnv != "" {
		return tokenEnv, nil
	}
	return "", fmt.Errorf("token was not supplied")
}

func main() {
	token := flag.String("token", "", "OAuth token")
	iamToken := flag.String("iamToken", "", "IAM token")
	tokenFile := flag.String("tokenFile", "", "Path to OAuth token")
	iamTokenFile := flag.String("iamTokenFile", "", "Path to IAM token")
	dryRun := flag.Bool("dryRun", false, "Simulate only, disables all mutable operations")
	metricsFile := flag.String("metricsFile", "", "Path to write metrics after the job is done")

	flag.Parse()
	plan := Plan(flag.Args())
	if len(plan) == 0 {
		log.Println("no plan in given in the command line, falling back to the default set")
		plan = []string{
			PreMainSolomon.Name,
			PreMainSolomonCloud.Name,
			ProdMainSolomonCloud.Name,
			ProdCloudSolomon.Name,
			PreCloudSolomon.Name,
		}
	}

	tokenStr, err := chooseToken(token, tokenFile, "SOLOMON_TOKEN")
	if err != nil {
		flag.Usage()
		os.Exit(1)
	}

	iamTokenStr := ""
	for _, name := range plan {
		if instance := instances[name]; instance.IAMBased {
			iamTokenStr, err = chooseToken(iamToken, iamTokenFile, "SOLOMON_IAM_TOKEN")
			if err != nil {
				log.Fatalln("cannot sync IAM based installations without token: %w", err)
			}
			break
		}
	}

	registry := solomon.NewRegistry(&solomon.RegistryOpts{UseNameTag: true})
	registry.Gauge("wakeTimestamp").Set(float64(time.Now().Unix()))

	client := SolomonClient{
		Token:    tokenStr,
		IAMToken: iamTokenStr,
		HTTP:     http.Client{Timeout: 30 * time.Second},
		DryRun:   *dryRun,
		Registry: registry,
		Juggler:  juggler.MakeClient("", tokenStr),
	}
	client.SyncGraphs(plan)
	client.SyncDashboards(plan)
	client.SyncMenu(plan)
	client.SyncNotificationChannels(plan)
	client.SyncAlerts(plan)

	if *metricsFile != "" {
		f, err := os.Create(*metricsFile)
		if err != nil {
			log.Fatalln("Failed to create file with metrics", err)
		}
		defer f.Close()
		_, _ = registry.StreamSpack(context.Background(), f, solomon.CompressionLz4)
	}
}
