package main

import (
	"log"
	"strings"
)

func (c *SolomonClient) ListNotificationChannelIds(instance *Instance) ([]string, error) {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/notificationChannels?pageSize=30"
	resp, err := c.Get(instance, endpoint)
	if err != nil {
		return nil, err
	}
	result := make([]string, 0)
	for {
		ids, nextToken, err := ParseListIdsWithToken(resp)
		if err != nil {
			return nil, err
		}
		result = append(result, ids...)
		if nextToken == "" {
			break
		}

		resp, err = c.Get(instance, endpoint+"&pageToken="+nextToken)
		if err != nil {
			return nil, err
		}
	}

	return result, nil
}

func (c *SolomonClient) GetNotificationChannel(instance *Instance, id string) (map[string]interface{}, error) {
	body, err := c.GetOptional(instance, "/api/v2/projects/"+instance.ProjectID+"/notificationChannels/"+id)
	if err != nil {
		return nil, err
	}
	if body == nil {
		return nil, nil
	}
	return deserialize(body)
}

func (c *SolomonClient) CreateNotificationChannel(instance *Instance, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/notificationChannels"
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Post(instance, endpoint, body)
}

func (c *SolomonClient) UpdateNotificationChannel(instance *Instance, id string, model map[string]interface{}) error {
	endpoint := "/api/v2/projects/" + instance.ProjectID + "/notificationChannels/" + id
	body, err := serialize(model)
	if err != nil {
		return err
	}
	return c.Put(instance, endpoint, body)
}

func (c *SolomonClient) DeleteNotificationChannel(instance *Instance, id string) error {
	return c.Delete(instance, "/api/v2/projects/"+instance.ProjectID+"/notificationChannels/"+id)
}

func (c *SolomonClient) SyncNotificationChannels(p Plan) {
	source, err := c.ListNotificationChannelIds(ProdMainSolomon)
	if err != nil {
		log.Println("Failed to ListNotificationChannelIds: " + err.Error())
		return
	}
	for _, id := range source {
		entity, err := c.GetNotificationChannel(ProdMainSolomon, id)
		if err != nil {
			log.Println("Failed to GetNotificationChannel " + id + ": " + err.Error())
			continue
		}
		if entity == nil {
			log.Println(ProdMainSolomon.Name + " NotificationChannel" + id + " already removed from source")
			continue
		}

		p.SyncInstances(func(instance *Instance) {
			if isIgnored(instance, entity) {
				log.Println(instance.Name + " NotificationChannel " + id + " SKIP marked as ignore on source")
			} else if instance.EnableChannel == OneToOne {
				c.syncNotificationChannel(instance, id, entity)
			}
		})
	}
	p.SyncInstances(func(instance *Instance) {
		c.deleteNotExistNotificationChannels(instance, source)
	})
}

func (c *SolomonClient) syncNotificationChannel(instance *Instance, id string, source map[string]interface{}) {
	patched := patchNotificationChannel(instance, source)
	target, err := c.GetNotificationChannel(instance, id)
	if err != nil {
		log.Println(instance.Name + " NotificationChannel " + id + " FAILED: " + err.Error())
		return
	}
	if target == nil {
		if err = c.CreateNotificationChannel(instance, patched); err == nil {
			log.Println(instance.Name + " NotificationChannel " + id + " CREATED")
		} else {
			log.Println(instance.Name + " NotificationChannel " + id + " FAILED: " + err.Error())
		}
	} else if isIgnored(instance, target) {
		log.Println(instance.Name + " NotificationChannel " + id + " SKIP because marked as ignore")
	} else {
		RemoveCommonAttributes(target)
		CopyVersion(target, patched)
		if isEqualLogDiff(instance, "channel", patched, target) {
			log.Println(instance.Name + " NotificationChannel " + id + " SKIP already sync")
		} else {
			if err = c.UpdateNotificationChannel(instance, id, patched); err == nil {
				log.Println(instance.Name + " NotificationChannel " + id + " UPDATED")
			} else {
				log.Println(instance.Name + " NotificationChannel " + id + " FAILED: " + err.Error())
			}
		}
	}
}

func (c *SolomonClient) deleteNotExistNotificationChannels(instance *Instance, ids []string) {
	unique := make(map[string]bool)
	if instance.EnableChannel == OneToOne {
		for _, id := range ids {
			unique[id] = true
		}
	}

	targetIds, err := c.ListNotificationChannelIds(instance)
	if err != nil {
		log.Println(instance.Name + " Failed to ListNotificationChannelIds: " + err.Error())
		return
	}
	for _, id := range targetIds {
		exist := unique[id]
		if !exist {
			entity, err := c.GetNotificationChannel(instance, id)
			if err != nil {
				log.Println(instance.Name + " NotificationChannel " + id + " FAILED: " + err.Error())
				continue
			}
			if entity == nil {
				continue
			} else if isIgnored(instance, entity) {
				log.Println(instance.Name + " NotificationChannel " + id + " SKIP because mark as ignore")
				continue
			}

			if err = c.DeleteNotificationChannel(instance, id); err == nil {
				log.Println(instance.Name + " NotificationChannel " + id + " DELETED")
			} else {
				log.Println(instance.Name + " NotificationChannel " + id + " FAILED: " + err.Error())
			}
		}
	}
}

func patchNotificationChannel(instance *Instance, source map[string]interface{}) map[string]interface{} {
	patched := applyMap(source, "", func(path string, s string) string {
		if s == "solomon" {
			return instance.ProjectID
		}
		patchedStr := s
		patchedStr = strings.ReplaceAll(patchedStr, "prestable", instance.Pre)
		patchedStr = strings.ReplaceAll(patchedStr, ProdMainSolomon.Host, instance.Host)
		if path == ".method.juggler.host" {
			patchedStr = strings.ReplaceAll(patchedStr, "solomon", instance.ProjectID)
		}
		return patchedStr
	})
	RemoveCommonAttributes(patched)
	return patched
}
