package main

import (
	"encoding/json"
	"regexp"
	"strings"
)

var graphParamRegexp = regexp.MustCompile("graph=([a-zA-Z0-9_-]+)")
var dashboardParamRegexp = regexp.MustCompile("dashboard=([a-zA-Z0-9_-]+)")

func RemoveCommonAttributes(source map[string]interface{}) {
	delete(source, "createdBy")
	delete(source, "updatedBy")
	delete(source, "createdAt")
	delete(source, "updatedAt")
}

func DropGeneratedID(source map[string]interface{}) {
	delete(source, "generatedId")
}

func Disclaimer(entityType string, patched map[string]interface{}) {
	maybeOld := patched["description"]
	var old string
	if maybeOld == nil {
		old = ""
	} else {
		old = maybeOld.(string)
	}
	header := "Synched from " + ProdMainSolomon.Host + "/admin/projects/solomon/" + entityType + "s/" + patched["id"].(string)
	if old != "" {
		patched["description"] = header + "\n\n" + old
	} else {
		patched["description"] = header
	}
}

func ApplyAllRules(instance *Instance, source string) string {
	if instance.ReplicaPatch == nil {
		return source
	}
	for _, rule := range instance.ReplicaPatch {
		source = rule.From.ReplaceAllString(source, rule.To)
	}
	return source
}

func PathReference(instance *Instance, source string) string {
	source = ApplyAllRules(instance, source)
	source = strings.ReplaceAll(source, ProdMainSolomon.Admin+"/staffOnly", instance.Admin+"/staffOnly")
	source = strings.ReplaceAll(source, ProdMainSolomon.AdminPre+"/staffOnly", instance.AdminPre+"/staffOnly")
	source = strings.ReplaceAll(source, ProdMainSolomon.Host, instance.Host)
	source = strings.ReplaceAll(source, "prestable", instance.Pre)
	source = strings.ReplaceAll(source, "project=solomon&", "project="+instance.ProjectID+"&")
	source = strings.ReplaceAll(source, "/solomon/", "/"+instance.ProjectID+"/")
	source = graphParamRegexp.ReplaceAllStringFunc(source, func(graph string) string {
		graph = strings.ReplaceAll(graph, "graph=", "")
		if graph == "auto" {
			return "graph=auto"
		}

		return "graph=" + instance.Prefix + graph
	})
	source = dashboardParamRegexp.ReplaceAllStringFunc(source, func(dashboard string) string {
		dashboard = strings.ReplaceAll(dashboard, "dashboard=", "")
		return "dashboard=" + instance.Prefix + dashboard
	})
	return source
}

func CopyVersion(source, target map[string]interface{}) {
	target["version"] = source["version"]
}

func ParseListIds(body []byte) ([]string, error) {
	type Item struct {
		ID string
	}

	type Items struct {
		Result []Item
	}

	var items Items
	err := json.Unmarshal(body, &items)
	if err != nil {
		return nil, err
	}

	r := make([]string, len(items.Result))
	for i, v := range items.Result {
		r[i] = v.ID
	}

	return r, nil
}

func ParseListIdsWithToken(body []byte) ([]string, string, error) {
	type Item struct {
		ID string
	}

	type Items struct {
		Items         []Item
		NextPageToken string `json:"nextPageToken"`
	}

	var items Items
	err := json.Unmarshal(body, &items)
	if err != nil {
		return nil, "", err
	}

	r := make([]string, len(items.Items))
	for i, v := range items.Items {
		r[i] = v.ID
	}

	return r, items.NextPageToken, nil
}
